#include <passport/infra/daemons/logstoreapi/src/utils/converter.h>
#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLogstoreApi;

Y_UNIT_TEST_SUITE(Converter) {
    Y_UNIT_TEST(PushSuccess) {
        NTest::TRequest req;
        req.Args = {
            {"file", "/path/to/file/file.log"},
            {"host", "test.yandex.net"},
            {"env", "testing"},
            {"stream_id", "721364981"},
            {"offset", "100500"},
        };
        req.RequestBody = "body";

        TPushRequest request = TRequestConverter::ConvertPushRequest(req);
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.File, "path/to/file/file.log");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.Host, "test.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.Env, "testing");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.StreamId, "721364981");
        UNIT_ASSERT_VALUES_EQUAL(request.Offset, 100500);
        UNIT_ASSERT_VALUES_EQUAL(request.Body, "body");
    }

    Y_UNIT_TEST(PushFail) {
        {
            NTest::TRequest req;
            req.RequestBody = "body";

            req.Args = {
                {"host", "test.yandex.net"},
                {"stream_id", "412348923"},
                {"offset", "100500"},
            };
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Required parameter 'file' is missing");
        }

        {
            NTest::TRequest req;
            req.Args = {
                {"file", "/path/to/file/file.log"},
                {"stream_id", "64738246"},
                {"offset", "100500"},
            };
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Required parameter 'host' is missing");
        }

        {
            NTest::TRequest req;
            req.Args = {
                {"file", "/path/to/file/file.log"},
                {"host", "test.yandex.net"},
                {"offset", "100500"},
            };
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Required parameter 'env' is missing");
        }

        {
            NTest::TRequest req;
            req.Args = {
                {"file", "/path/to/file/file.log"},
                {"host", "test.yandex.net"},
                {"env", "testing"},
                {"offset", "100500"},
            };
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Required parameter 'stream_id' is missing");
        }

        {
            NTest::TRequest req;
            req.Args = {
                {"file", "/path/to/file/file.log"},
                {"host", "test.yandex.net"},
                {"env", "testing"},
                {"stream_id", "123"},
                {"offset", "asf"},
            };
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Parameter 'offset' is invalid");
            req.Args.erase("offset");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TRequestConverter::ConvertPushRequest(req),
                TBadRequestError,
                "Required parameter 'offset' is missing");
        }
    }

    Y_UNIT_TEST(StreamSuccess) {
        NTest::TRequest req;
        req.Args = {
            {"file", "/path/to/file/file.log"},
            {"host", "test.yandex.net"},
            {"env", "testing"},
            {"stream_id", "721364981"},
            {"create_time", "721364981"},
            {"offset", "100500"},
        };
        req.RequestBody = "body";

        TStreamRequest request = TRequestConverter::ConvertStreamRequest(req);
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.File, "path/to/file/file.log");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.Host, "test.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.Env, "testing");
        UNIT_ASSERT_VALUES_EQUAL(request.StreamInfo.StreamId, "721364981");
        UNIT_ASSERT_VALUES_EQUAL(request.Timestamp, TInstant::Seconds(721364981));
    }

    Y_UNIT_TEST(StreamFail) {
        NTest::TRequest req;
        req.Args = {
            {"file", "/path/to/file/file.log"},
            {"host", "test.yandex.net"},
            {"env", "testing"},
            {"stream_id", "721364981"},
            {"create_time", "721364981sf"},
            {"offset", "100500"},
        };
        req.RequestBody = "body";

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TRequestConverter::ConvertStreamRequest(req),
            TBadRequestError,
            "Parameter 'create_time' is invalid");
        req.Args.erase("create_time");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TRequestConverter::ConvertStreamRequest(req),
            TBadRequestError,
            "Required parameter 'create_time' is missing");
    }
}
