#include "common.h"

#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>
#include <passport/infra/daemons/logstoreapi/src/utils/tz_msk.h>
#include <passport/infra/daemons/logstoreapi/src/writers/logs_manager.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

using namespace NPassport;
using namespace NPassport::NLogstoreApi;

Y_UNIT_TEST_SUITE(Logsmanager) {
    Y_UNIT_TEST(isDeadlineReached) {
        TLogsManager::TWriterWrapper w{
            .Deadline = TInstant::Seconds(10005000),
        };

        UNIT_ASSERT(!w.IsDeadlineReached(TInstant::Seconds(10000000)));
        UNIT_ASSERT(w.IsDeadlineReached(TInstant::Seconds(10005000)));
        UNIT_ASSERT(w.IsDeadlineReached(TInstant::Seconds(10009000)));
    }

    static TString FetchFilename(TWriterPtr w) {
        return static_cast<TNamedWriter*>(w.get())->Filename;
    }

    Y_UNIT_TEST(getLogWriterErrors) {
        SetTzMsk();
        const TInstant now = TInstant::ParseIso8601("2021-09-29T22:18:34.000000+03");

        TLogsManager manager(
            "/kek",
            std::make_shared<TNamedWriterFactory>(),
            std::vector<TString>{"testing", "production"});

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            manager.GetLogWriter("foo", "/some/file.log", ETimeAggregation::Day, now),
            TBadRequestError,
            "Env is unknown: 'foo'");
        UNIT_ASSERT_EXCEPTION(
            manager.GetLogWriter("testing", "/some/file.log", ETimeAggregation::Day, now),
            TIoException);
    }

    Y_UNIT_TEST(getLogWriterSame) {
        SetTzMsk();
        const TInstant now = TInstant::ParseIso8601("2021-09-29T22:18:34.000000+03");

        TLogsManager manager(
            "/kek",
            std::make_shared<TNamedWriterFactory>(),
            std::vector<TString>{"testing", "production"});

        TWriterPtr w = manager.GetLogWriter("testing", "some/file.log", ETimeAggregation::Day, now);
        UNIT_ASSERT_VALUES_EQUAL(
            "/kek/testing/some/file.log.20210929",
            FetchFilename(w));

        TWriterPtr w2 = manager.GetLogWriter("testing", "some/file.log", ETimeAggregation::Day, now + TDuration::Hours(1));
        UNIT_ASSERT_VALUES_EQUAL(
            "/kek/testing/some/file.log.20210929",
            FetchFilename(w2));

        UNIT_ASSERT_EQUAL(w, w2);
    }

    Y_UNIT_TEST(getLogWriterNext) {
        SetTzMsk();

        TLogsManager manager(
            "/kek",
            std::make_shared<TNamedWriterFactory>(),
            std::vector<TString>{"testing", "production"});

        struct TCase {
            TString Name;
            TInstant Now;
            ETimeAggregation Aggr = ETimeAggregation::Hour;
            TString Filename;
        };

        std::vector<TCase> cases = {
            TCase{
                .Name = "#1. log for 22:00",
                .Now = TInstant::ParseIso8601("2021-09-29T22:18:34.000000+03"),
                .Aggr = ETimeAggregation::Hour,
                .Filename = "/kek/testing/file.log.2021092922",
            },
            TCase{
                .Name = "#2. log for 23:00",
                .Now = TInstant::ParseIso8601("2021-09-29T23:18:34.000000+03"),
                .Aggr = ETimeAggregation::Hour,
                .Filename = "/kek/testing/file.log.2021092923",
            },
            TCase{
                .Name = "#3. log for 23:00 - period still actual",
                .Now = TInstant::ParseIso8601("2021-09-29T23:20:34.000000+03"),
                .Aggr = ETimeAggregation::Day,
                .Filename = "/kek/testing/file.log.2021092923",
            },
            TCase{
                .Name = "#4. log for 2021-09-30",
                .Now = TInstant::ParseIso8601("2021-09-30T00:18:34.000000+03"),
                .Aggr = ETimeAggregation::Day,
                .Filename = "/kek/testing/file.log.20210930",
            },
            TCase{
                .Name = "#5. log for 2021-09-30",
                .Now = TInstant::ParseIso8601("2021-09-30T03:18:34.000000+03"),
                .Aggr = ETimeAggregation::Hour,
                .Filename = "/kek/testing/file.log.20210930",
            },
        };

        for (const TCase& c : cases) {
            UNIT_ASSERT_VALUES_EQUAL_C(
                c.Filename,
                FetchFilename(manager.GetLogWriter("testing", "file.log", c.Aggr, c.Now)),
                c.Name);
        }
    }

    Y_UNIT_TEST(getLogWriterCleanup) {
        SetTzMsk();
        const TInstant now = TInstant::ParseIso8601("2021-09-29T22:18:34.000000+03");

        class TTestManager: public TLogsManager {
        public:
            using TLogsManager::Cleanup;
            using TLogsManager::TLogsManager;
        };

        TTestManager manager(
            "/kek",
            std::make_shared<TNamedWriterFactory>(),
            std::vector<TString>{"testing", "production"});

        manager.GetLogWriter("testing", "some/file.log", ETimeAggregation::Day, now);
        manager.GetLogWriter("testing", "some/file.log", ETimeAggregation::Day, now + TDuration::Hours(3));
        UNIT_ASSERT_VALUES_EQUAL(1, manager.Cleanup(now));
        UNIT_ASSERT_VALUES_EQUAL(0, manager.Cleanup(now + TDuration::Hours(3)));
        UNIT_ASSERT_VALUES_EQUAL(1, manager.Cleanup(now + TDuration::Hours(24 + 3)));
        UNIT_ASSERT_VALUES_EQUAL(0, manager.Cleanup(now + TDuration::Hours(48 + 3)));
    }
}
