#include "common.h"

#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>
#include <passport/infra/daemons/logstoreapi/src/writers/sequencer.h>

#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLogstoreApi;

std::shared_ptr<NLb::TResourceDispatcher> MakeResourceDispatcher(size_t memoryLimit = 1) {
    std::shared_ptr<NLb::TResourceDispatcher> resourceDispatcher = std::make_shared<NLb::TResourceDispatcher>(TSequencer::RESOURCE_ID);
    resourceDispatcher->AddResource(TSequencer::RESOURCE_ID, memoryLimit, 0);
    return resourceDispatcher;
}

class TTestSequencer: TSequencer {
public:
    explicit TTestSequencer(size_t bufferSize = 4096, size_t memoryLimit = 1 << 20)
        : TSequencer(
              TSequencerSettings{
                  .BufferSize = bufferSize,
              },
              MakeResourceDispatcher(memoryLimit),
              std::make_shared<TSequencerUnistatCtx>("some_file"))
    {
    }

    using TSequencer::Pop;
    using TSequencer::Push;
    using TSequencer::Reset;
    using TSequencer::Write;

    using TSequencer::FlushBuffer;

    using TSequencer::Contiguous;
    using TSequencer::Overlapping;
};

Y_UNIT_TEST_SUITE(Sequencer) {
    Y_UNIT_TEST(Simple) {
        TTestSequencer s;
        TFakeWriter w;

        UNIT_ASSERT_VALUES_EQUAL(s.Contiguous(), false);
        UNIT_ASSERT_VALUES_EQUAL(s.Overlapping(), false);

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "asdf1", "asdf2", "asdf3",
                       }, 0));

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "asdf4", "asdf5",
                   }, 18), NThreading::NewPromise());

        UNIT_ASSERT_VALUES_EQUAL(s.Contiguous(), true);
        UNIT_ASSERT_VALUES_EQUAL(s.Overlapping(), false);

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "asdf4", "asdf5",
                   }, 17), NThreading::NewPromise());

        UNIT_ASSERT_VALUES_EQUAL(s.Contiguous(), false);
        UNIT_ASSERT_VALUES_EQUAL(s.Overlapping(), true);

        UNIT_ASSERT_VALUES_EQUAL(s.Pop().first->Offset(), 17);
        UNIT_ASSERT_VALUES_EQUAL(s.Pop().first->Offset(), 18);
    }

    Y_UNIT_TEST(Flush) {
        TTestSequencer s;
        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "1", "2", "3",
                   }), NThreading::NewPromise());

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host", {
                                      "5", "6",
                                  }, 8), NThreading::NewPromise());

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "4",
                   }, 6), NThreading::NewPromise());

        TFakeWriter w;
        s.FlushBuffer(w, false);
        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "1\n2\n3\n4\n5\n6\n");
        UNIT_ASSERT_VALUES_EQUAL(w.Offset(), 12);
    }

    Y_UNIT_TEST(FlushForce) {
        TTestSequencer s;

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "1", "2", "3",
                   }, 1), NThreading::NewPromise());

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "5", "6",
                   }, 12), NThreading::NewPromise());

        s.Push(MakeChunk(
                   "unused file id",
                   "unused host",
                   {
                       "4",
                   }, 8), NThreading::NewPromise());

        TFakeWriter w;
        s.FlushBuffer(w, false);

        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "");
        UNIT_ASSERT_VALUES_EQUAL(w.Offset(), 0);

        s.FlushBuffer(w, true);

        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "1\n2\n3\n4\n5\n6\n");
        UNIT_ASSERT_VALUES_EQUAL(w.Offset(), 16);
    }

    Y_UNIT_TEST(Sequential) {
        TTestSequencer s;
        TFakeWriter w;

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "1", "2", "3",
                       }));

        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "1\n2\n3\n");

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "5", "6",
                       }, 8));

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "5", "6",
                       }, 8));

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "5", "6",
                       }, 8));

        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "1\n2\n3\n");

        s.Write(w, MakeChunk(
                       "unused file id",
                       "unused host",
                       {
                           "4",
                       }, 6));

        UNIT_ASSERT_VALUES_EQUAL(w.Output(), "1\n2\n3\n4\n5\n6\n");
        UNIT_ASSERT_VALUES_EQUAL(w.Offset(), 12);

        UNIT_ASSERT_EXCEPTION_CONTAINS(s.Write(w, MakeChunk(
                                                      "unused file id",
                                                      "unused host",
                                                      {
                                                          "4",
                                                      }, 1)), TDuplicateError, "Chunk arrived too late");

        UNIT_ASSERT_NO_EXCEPTION(s.Reset(w, 0, EMissingChunkPolicy::ForceFlush));
        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "unused file id",
                                                "unused host",
                                                {
                                                    "4",
                                                }, 0)));
    }

    Y_UNIT_TEST(BufferOverflow) {
        TTestSequencer s(1);
        TFakeWriter w;

        UNIT_ASSERT_EXCEPTION_CONTAINS(s.Write(w, MakeChunk(
                                                      "unused file id",
                                                      "unused host",
                                                      {
                                                          "1", "2", "3",
                                                      }, 8096)), TBufferOverflowError, "Chunk arrived too early");
    }

    Y_UNIT_TEST(QueueOverflow) {
        TTestSequencer s(1 << 10, 10);
        TFakeWriter w;

        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "unused file id",
                                                "unused host",
                                                {
                                                    "a",
                                                }, 1)));

        UNIT_ASSERT_EXCEPTION_CONTAINS(s.Write(w, MakeChunk(
                                                      "unused file id",
                                                      "unused host",
                                                      {
                                                          "asdfasjdhfblasjdhfblsd",
                                                      }, 10)), TBufferOverflowError, "Memory limit reached, please try backing off");
    }

    Y_UNIT_TEST(QueueOverflow2) {
        TTestSequencer s(1 << 10, 10);
        TFakeWriter w;

        std::vector<TString> data;
        data.reserve(1 << 20);
        for (size_t i = 0; i < 1 << 20; ++i) {
            data.push_back("asdf");
        }

        // This is normal behaviour for LB, in our case we rely on chunk size being smaller than limit
        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "unused file id",
                                                "unused host",
                                                std::move(data), 1)));

        UNIT_ASSERT_EXCEPTION_CONTAINS(s.Write(w, MakeChunk(
                                                      "unused file id",
                                                      "unused host",
                                                      {
                                                          "",
                                                      }, 10)), TBufferOverflowError, "Memory limit reached, please try backing off");
    }

    Y_UNIT_TEST(LinesMisaligned) {
        TTestSequencer s;
        TFakeWriter w;

        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "some file",
                                                "some host",
                                                {
                                                    "some OK log line",
                                                }, 0)));

        UNIT_ASSERT_NO_EXCEPTION(s.Reset(w, 0, EMissingChunkPolicy::Discard));
        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "some file",
                                                "some host",
                                                {
                                                    "misaligned log line",
                                                }, 0)));

        UNIT_ASSERT_NO_EXCEPTION(s.Write(w, MakeChunk(
                                                "some file",
                                                "some host",
                                                {
                                                    "OK continuation",
                                                }, 20)));

        UNIT_ASSERT_EXCEPTION_CONTAINS(s.Write(w, MakeChunk(
                                                      "some file",
                                                      "some host",
                                                      {
                                                          "misaligned continuation",
                                                      }, 20)), TBadRequestError, "Log lines appear to be misaligned");

        UNIT_ASSERT_VALUES_EQUAL(
            w.Output(),
            "some OK log line\n"
            "ERROR: Corrupted stream: log lines appear to be misaligned\n"
            "misaligned log line\n"
            "OK continuation\n");
    }
}
