#include "sids.h"
// DO_NOT_STYLE

#include <passport/infra/daemons/mda/library/common_module.h>
#include <passport/infra/daemons/mda/library/util.h>
#include <library/cpp/blackbox2/blackbox2.h>

#include <passport/infra/libs/cpp/xml/config.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/fs.h>

#include <algorithm>
#include <cstring>
#include <iostream>
#include <sstream>
#include <stdio.h>

namespace NPassport::NMda {

static const TString emptyStr;
static const TString dbpoolstatsStr("dbpoolstats");

//
// Helper escaping UTF-8 characters into JavaScript \unnnn symbols.
// This doesn't verify validity of UTF-8 encoding, we assume
// input string is a valid one, so we just change its
// presentation
//
static TString UtfEscape(const TString& s, bool quote_only) {
    char buf[sizeof("\\u0000")];
    TString ret;
    ret.reserve(s.size() * 3);
    for (TString::const_iterator it = s.begin(); it != s.end(); ++it) {
        unsigned long c;
        unsigned long cur = (unsigned char)*it;
        if (quote_only || cur <= 0x7F) {
            if (*it == '\\' || *it == '"' || *it == '/') {
                ret.push_back('\\');
            }
            ret.push_back(*it);
            continue;
        }
        if (cur >= 0xC2 && cur <= 0xDF) {
            // two-byte char
            c = (cur & 0x1F) << 6;
            cur = (unsigned long)(unsigned char)*(++it);
            c += cur & 0x3F;
        } else if (cur >= 0xE0 && cur <= 0xEF) {
            // three-byte char
            c = (cur & 0x0F) << 12;
            cur = (unsigned long)(unsigned char)*(++it);
            c += (cur & 0x3F) << 6;
            cur = (unsigned long)(unsigned char)*(++it);
            c += cur & 0x3F;
        } else {
            // four-byte char
            c = (cur & 0x07) << 18;
            cur = (unsigned long)(unsigned char)*(++it);
            c += (cur & 0x3F) << 12;
            cur = (unsigned long)(unsigned char)*(++it);
            c += (cur & 0x3F) << 6;
            cur = (unsigned long)(unsigned char)*(++it);
            c += cur & 0x3F;
        }
        snprintf(buf, sizeof(buf), "\\u%04lX", c);
        ret.append(buf);
    }

    return ret;
}

//
// Static memebers
//
const std::regex Sids::callbackRegex_("^[[:alnum:]_-]+(\\.[[:alnum:]_-]+)*$");

static const TString nagiosOkStr("\r\n<html><head><title>MDA sids monitoring status</title></head><body><h2>OK</h2></body></html>\r\n");
static const TString nagiosBlackboxErrStr("\r\n<html><head><title>MDA sids monitoring status</title></head><body><h2>Error: blackbox not available</h2></body></html>\r\n");
static const TString nagiosUnconstructedStr("\r\n<html><head><title>MDA sids monitoring status</title></head><body><h2>Module is unconstructed</h2></body></html>\r\n");
static const TString nagiosForcedDownStr("\r\n<html><head><title>MDA sids monitoring status</title></head><body><h2>Externally forced out-of-service</h2></body></html>\r\n");

const TString callbackCgiStr_("callback");
const TString yuArgStr_("yu");
const TString yuCookieStr_("yandexuid");

const TString loginPatternStr_("$LOGIN");

const TString listQuery1 = "SELECT * FROM ((SELECT a.login,a.ena,a.glogout,su.sid,su.suid,su.login as login2,ui.display_name FROM subscription su, accounts a, userinfo ui WHERE su.uid=a.uid AND a.uid=ui.uid AND a.uid=";
const TString listQuery2 = ") UNION ( SELECT '','','',sid,login_rule,login,'' FROM subscription WHERE uid=";
const TString listQuery3 = " AND sid IN (668,669))) aa";

static const TString defaultRootDomain = "yandex.ru";
static const TString moikrugDomain = "moikrug.ru";

void Sids::Serialize(const NSezamApi::TPersonalInfo& info, TString& buf, bool escape) {
    buf.append("\"uid\": \"").append(info.Uid).append("\",\n");
    buf.append("\"login\": \"").append(info.Login).append("\",\n");
    buf.append("\"displayName\": {\"name\": \"").append(UtfEscape(info.DisplayName, !escape));
    buf.append("\", \"default_avatar\": \"").append(info.DefaultAvatar).push_back('"');

    if (info.Social) {
        buf.append(", \"social\": {\"profileId\": ").append(info.ProfileId);
        buf.append(", \"provider\": \"").append(info.Provider).append("\"}");
    }
    buf.append("},\n");
    buf.append("\"defaultEmail\": \"").append(info.DefaultEmail).append("\",\n");
    buf.append("\"attributes\": {");
    for (NSezamApi::TPersonalInfo::TAttrMap::const_iterator it = info.Attributes.begin(), beg = it;
         it != info.Attributes.end();
         ++it) {
        if (it != beg) {
            buf.append(", ");
        }
        buf.append("\"").append(it->first).append("\": ");

        if (const TString* val = std::get_if<TString>(&it->second); val) {
            buf.push_back('"');
            buf.append(*val);
            buf.push_back('"');
        } else if (const bool* val = std::get_if<bool>(&it->second); val) {
            buf.append(*val ? "true" : "false");
        }
    }
    buf.push_back('}');
}

Sids::Sids()
    : 
     DbCtx_(std::make_shared<NDbPool::TDbPoolCtx>())
     {
}

Sids::~Sids() = default;

void Sids::Init(const NXml::TConfig& config) {
    try {
        TString componentXPath = NUtils::CreateStr(
            config.Contains("/config") ? "/config" : "/fastcgi",
            R"(/components/component[@name="sids"])");

        config.InitCommonLog(componentXPath + "/logger_common");
        TLog::Info("Starting sids passport module");

        DbCtx_->InitLogger(config.AsString(componentXPath + "/log_dbpool"));

        InitTvm(config, componentXPath + "/tvm");

        TString bb_host = config.AsString(componentXPath + "/blackbox/db_host", "");
        if (bb_host.empty()) {
            throw std::runtime_error("blackbox pool not configured");
        }

        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            TvmClient_,
            NTvmCommon::TServiceTickets::BLACKBOX_)};

        try {
            BlackboxDb_ = DbCtx_->CreateDb(config, componentXPath + "/blackbox", std::move(opts));
        } catch (const NDbPool::TException& e) {
            TLog::Info("Sids module: failed to initialize blackbox DB connection: %s, exitting!", e.what());
            exit(1);
        }
        TLog::Info() << "Sids module: blackbox pool configured, using "
                     << BlackboxDb_->GetDbInfo();

        InitServeDomains(&config, componentXPath);

        const TString commonXpath = NUtils::CreateStr(
            config.Contains("/config") ? "/config" : "/fastcgi",
            R"(/components/component[@name="common-module"])");

        RootKeyspace_ = config.AsString(commonXpath + "/root_domain/keyspace", "yandex.ru");
        ForceDownFile_ = config.AsString(commonXpath + "/force_down_file", "");

        // yandexuid csrf check
        TString enableyu = config.AsString(componentXPath + "/enable_yandexuid_check", "no");
        EnableYandexuidCheck_ = enableyu == "yes" || enableyu == "1";
        TLog::Info("Sids module: yandexuid check %s", EnableYandexuidCheck_ ? "enabled" : "disabled");

        AccountsSettings_ = std::make_unique<NSezamApi::TAccountsSettings>(config, componentXPath + "/accounts_settings");

        InitFailed_ = false;
        TLog::Info("Sids passport module initialized");
    } catch (const std::exception& e) {
        TLog::Error("Sids passport module init failed: %s", e.what());
    } catch (...) {
        TLog::Error("Sids passport module init failed: unknown exception");
    }
}

void Sids::HandleRequest(NPassport::NCommon::TRequest& request) {
    request.ScanCgiFromBody();
    TString path = request.GetPath();

    if (path.compare(0, 7, "/nagios") == 0 ||
        path.compare(0, 16, "/services/nagios") == 0) {
        ProcessNagios(&request);
        return;
    }

    NCommon::TReqCtx ctx(&request);

    if (path.compare(0, 9, "/accounts") == 0) {
        ProcessAccounts(ctx);
        return;
    }
}

void Sids::AddUnistat(NUnistat::TBuilder& builder) {
    DbCtx_->AddUnistat(builder);
}

bool Sids::InitFailed() const {
    return InitFailed_;
}

std::pair<TString, TString> Sids::GetDomainInfo(const TString& host) {
    ServeDomainsCIter domain = GetServeDomain(NCommon::Utils::HostFromUri(host));
    bool found = ServeDomains_.end() != domain;
    TString domain_str = found ? domain->first : defaultRootDomain;
    const TString& keyspace = found ? domain->second : RootKeyspace_;
    if (!domain_str.empty() && domain_str[0] == '.') {
        domain_str.erase(0, 1);
    }
    if (moikrugDomain == domain_str) {
        domain_str.assign(defaultRootDomain);
    }
    return std::make_pair(keyspace, domain_str);
}

Sids::ServeDomainsCIter Sids::GetServeDomain(const TString& host) {
    ServeDomainsCIter domain = ServeDomains_.end();
    TString::size_type ldotpos = host.rfind('.');
    TString::size_type dotpos = 0;
    while (dotpos < ldotpos) {
        domain = ServeDomains_.find(host.substr(dotpos));
        if (domain != ServeDomains_.end()) {
            break;
        }
        if (dotpos == 0) {
            TString dotthost(1, '.');
            dotthost.append(host);
            domain = ServeDomains_.find(dotthost);
            if (domain != ServeDomains_.end()) {
                break;
            }
        }
        dotpos = host.find('.', dotpos + 1);
    }
    return domain;
}

bool Sids::CheckCsrf(NCommon::TReqCtx& ctx, const TString& sess_cookie) const {
    // check CSRF protection if enabled
    if (!EnableYandexuidCheck_) {
        return true;
    }

    const TString yu_arg = NCommon::CommonModule::GetArg(ctx, yuArgStr_);
    TString yu_cookie;

    if (!ctx.Cookies.Get(yuCookieStr_, yu_cookie) || yu_arg.empty() || yu_cookie != yu_arg) {
        TString referer = ctx.Referer;
        if (referer.size() > 100) {
            referer.resize(100);
        }

        TString url = ctx.Request->GetUrl();
        if (url.size() > 100) {
            url.resize(100);
        }

        TLog::Error("Error: SidsModule CSRF check failed: yandexuid mismatch. Request <%s>, yandexuid cookie <%s> Session_id cookie <%s> referrer <%s>",
                   url.c_str(),
                   yu_cookie.c_str(),
                   sess_cookie.c_str(),
                   referer.c_str());

        return false;
    }

    return true;
}

static const TString TITLE = "MDA sids monitoring status";
void Sids::ProcessNagios(NPassport::NCommon::TRequest* request) {
    HttpCodes status;
    TString ret;

    if (InitFailed()) {
        ret.assign(nagiosUnconstructedStr);
        TLog::Error("Fatal: /nagios error (unconstructed sids module)");
        status = HTTP_INTERNAL_SERVER_ERROR;
    } else if (request->HasArg(dbpoolstatsStr)) {
        NDbPool::TDbPoolStats stats(TITLE);
        stats.Add(BlackboxDb_.get());
        ret.assign(stats.Result());
        status = HTTP_OK;
    } else if (!IsForceDown()) {
        TString msg;
        if (!BlackboxDb_ || !BlackboxDb_->IsOk(&msg)) {
            ret.assign(nagiosBlackboxErrStr);
            TLog::Error("sids nagios error: blackbox not available");
            status = HTTP_INTERNAL_SERVER_ERROR;
        } else {
            ret.assign(nagiosOkStr);
            status = HTTP_OK;
        }
    } else {
        ret.assign(nagiosForcedDownStr);
        TLog::Error("sids nagios error: externally forced out-of-service");
        status = HTTP_SERVICE_UNAVAILABLE;
    }

    request->SetStatus(status);
    request->SetHeader("Content-Type", "text/html");
    request->Write(ret);
}

void Sids::ProcessAccounts(NCommon::TReqCtx& ctx) {
    // Use this in error messages
    TString sess_cookie;
    TString callback;

    // Send this if everything is OK
    std::unique_ptr<NSezamApi::TAccounts> accounts;

    try {
        callback = NCommon::CommonModule::GetArg(ctx, callbackCgiStr_);
        if (!callback.empty() && !std::regex_match(callback.cbegin(), callback.cend(), callbackRegex_)) {
            TLog::Error("Warning: SidsModule/accounts: bad callback argument <%s>, Session_id <%s>, returning empty list",
                       NUtils::EscapeEol(callback).c_str(),
                       sess_cookie.c_str());
            callback.clear();
            SendAccountsList(ctx, callback);
            return;
        }

        // Do nothing if there is no session at all
        if (!ctx.Cookies.Get("Session_id", sess_cookie) || sess_cookie.empty()) {
            SendAccountsList(ctx, callback);
            return;
        }

        // If failed to initialize return empty list
        if (InitFailed()) {
            TLog::Error("Error: SidsModule: have not been initialized properly, return empty list, Session_id <%s>",
                       sess_cookie.c_str());
            SendAccountsList(ctx, callback);
            return;
        }

        if (!CheckCsrf(ctx, sess_cookie)) {
            SendAccountsList(ctx, callback);
            return;
        }

        std::pair<TString, TString> res = GetDomainInfo(ctx.Host);

        // Finally, fetch list from the db or blackbox
        accounts = NSezamApi::GetAccountsFromBb(*BlackboxDb_,
                                                *AccountsSettings_,
                                                {
                                                    sess_cookie,
                                                    {},
                                                    res.second,
                                                    ctx.Userip,
                                                });
    } catch (const std::exception& e) {
        TLog::Error("Error: exception in SidsModule/accounts: %s, Session_id <%s>, returning empty list",
                   e.what(),
                   sess_cookie.c_str());
    } catch (...) {
        TLog::Error("Error: unknown exception in SidsModule/accounts: Session_id <%s>, returning empty list",
                   sess_cookie.c_str());
    }

    SendAccountsList(ctx, callback, accounts.get());
}

void Sids::InitTvm(const NXml::TConfig& config, const TString& path) {
    NTvmAuth::NTvmApi::TClientSettings tvmSettings;
    tvmSettings.SetSelfTvmId(config.AsNum<ui32>(path + "/self_tvmid"));
    tvmSettings.SetDiskCacheDir(config.AsString(path + "/disk_cache_dir"));
    tvmSettings.EnableServiceTicketsFetchOptions(
            NUtils::ReadFile(config.AsString(path + "/secret_path")),
            {{NTvmCommon::TServiceTickets::BLACKBOX_, config.AsNum<ui32>(path + "/bb_tvmid")}});

    TvmClient_ = std::make_shared<NTvmAuth::TTvmClient>(tvmSettings, NTvmLogger::TLogger::Create());
}

void Sids::InitServeDomains(const NPassport::NXml::TConfig* config,
                            const TString& componentXPath) {
    std::vector<TString> domains = config->SubKeys(componentXPath + "/serve_domains/domain");

    for (const TString& d : domains) {
        TString domain_id = config->AsString(d + "/@id");
        if (domain_id.empty() || domain_id[0] != '.') {
            std::stringstream str;
            str << "Sids cannot serve domain " << domain_id << " (must start with '.')";
            throw std::runtime_error(str.str());
        }
        TString keyspace = config->AsString(d + "/keyspace", "!none!");
        if ("!none!" == keyspace) {
            throw std::runtime_error("Bad keyspace value at key: " + keyspace);
        }
        if (ServeDomains_.find(domain_id) != ServeDomains_.end()) {
            throw std::runtime_error("Duplicate domain id value at key: " + keyspace);
        }

        TLog::Info("Adding domain: %s", domain_id.c_str());
        ServeDomains_.insert(std::make_pair(domain_id, keyspace));
    }
}

bool Sids::IsForceDown() const {
    return !ForceDownFile_.empty() && NFs::Exists(ForceDownFile_);
}

void Sids::SendAccountsList(NCommon::TReqCtx& ctx, const TString& callback, const NSezamApi::TAccounts* accounts) {
    TString body;
    body.reserve(4096);
    if (!callback.empty()) {
        body.assign(callback).push_back('(');
    }

    body.push_back('{');
    if (accounts) {
        body.append("\"accounts\": [");
        for (const NSezamApi::TPersonalInfo& p : accounts->Users) {
            if (p.Uid.empty()) {
                continue;
            }
            body.append("\n{");
            Serialize(p, body, false);
            body.append("},");
        }
        if (body.back() == ',') {
            body.pop_back();
        }
        body.append("\n],\n\"default_uid\": \"").append(accounts->DefaultUid);
        body.append("\",\n\"can-add-more\": ").append(accounts->CanMore ? "true" : "false");
    }
    body.push_back('}');

    if (!callback.empty()) {
        body.push_back(')');
    }
    ctx.Request->SetStatus(HTTP_OK);
    ctx.Request->SetHeader("Content-Type", "text/javascript; charset=utf-8");
    ctx.Request->Write(body);
}

}
