#include "factors_by_number_processor.h"

#include "aggregates_helper.h"
#include "exceptions.h"
#include "number_meta_helper.h"
#include "runtime_context.h"
#include "utils.h"

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/geobase/geobase.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NPharmaApi {
    TFactorsByNumberProcessor::TFactorsByNumberProcessor(const TRuntimeContext& runtime)
        : TFactorsByNumberProcessor(
              runtime.GetDb(),
              runtime.GetGeobase(),
              runtime.GetFactorsSettings(),
              runtime.GetTestingNumbers())
    {
    }

    TFactorsByNumberProcessor::TFactorsByNumberProcessor(NDbPool::TDbPool& db,
                                                         const NGeobase::IGeobase& geobase,
                                                         const TFactorsSettings settings,
                                                         const TTestingNumbers& testingNumbers)
        : Db_(db)
        , Geobase_(geobase)
        , Settings_(settings)
        , TestingNumbers_(testingNumbers)
    {
    }

    static const TString PHONE_NUMBER = "phone_number";

    TFactorsByNumberResult TFactorsByNumberProcessor::Process(const NCommon::TRequest& request) {
        const TString& rawNumber = request.GetArg(PHONE_NUMBER);
        if (rawNumber.empty()) {
            throw TInvalidValueError() << "missing arg '" << PHONE_NUMBER << "'";
        }
        const TString number = TUtils::SanitizeNumber(rawNumber);

        if (TestingNumbers_ && !TestingNumbers_->contains(number)) {
            throw TInvalidValueError() << "this '" << PHONE_NUMBER << "' is not allowed in testing";
        }

        TFactorsByNumberResult res{
            .Factors = CalculateFactors(FetchFromDb(number), Settings_, Geobase_),
        };

        try {
            res.NumberMeta = TNumberMetaHelper::Process(number);
        } catch (const std::exception& e) {
            TLog::Debug() << "FactorsByNumber: Failed to get number meta: " << e.what();
        }

        return res;
    }

    NDbPool::TTable TFactorsByNumberProcessor::FetchFromDb(const TString& number) const {
        NDbPool::TBlockingHandle handle(Db_);

        const TString escapedNumber = handle.EscapeQueryParam(number);
        const TString query =
            NUtils::CreateStr(
                "(",
                "SELECT ts, NULL as uid, ip, ua, type FROM usages WHERE phonenum=", escapedNumber,
                " ORDER BY ts LIMIT ", Settings_.Limit,
                ") UNION ALL (",
                "SELECT add_ts as 'ts', uid, ip, ua, type FROM bindings WHERE phonenum=", escapedNumber,
                " ORDER BY add_ts LIMIT ", Settings_.Limit,
                ")");

        return handle.Query(query)->ExctractTable();
    }

    TFactorsByNumberResult::TFactors TFactorsByNumberProcessor::CalculateFactors(NDbPool::TTable table, const TFactorsSettings& settings, const NGeobase::IGeobase& geobase,
                                                                                 const time_t now) {
        if (table.empty()) {
            return {};
        }

        const size_t reserve = table.size();

        TTotalAggregator usages;
        TTotalAggregator bindins;

        TUniqueAggregator<ui64> uidAggr(reserve);
        TUniqueAggregator<TString> ipAggr(reserve);
        TUniqueAggregator<std::string> asAggr(reserve);
        TUniqueAggregator<TString> uaAggr(reserve);

        std::unordered_map<TString, std::string> asCache;
        asCache.reserve(reserve);

        for (const NDbPool::TRow& row : table) {
            time_t ts = 0;
            std::optional<ui64> uid;
            TString ip;
            TString userAgent;
            ui32 type = 0;

            row.Fetch(ts, uid, ip, userAgent, type);

            // inverted ts allowes to cheeply reverse order of events
            const time_t delta = now - TUtils::InvertTs(ts);

            switch (type) {
                case 1:
                    usages.Add(delta);
                    break;
                case 2:
                    bindins.Add(delta);
                    break;
                default:
                    TLog::Error() << "FactorsByNumber: unsupported event type: " << type;
                    continue;
            }

            if (type == 2) {
                // bindings
                Y_ENSURE(uid, "internal error: missing uid for binding: ts=" << ts);
                uidAggr.Add(delta, *uid);
            }

            if (ip) {
                ipAggr.Add(delta, ip);

                auto it = asCache.find(ip);
                if (it == asCache.end()) {
                    std::optional<std::string> as = geobase.GetAs(ip);
                    asCache.emplace(ip, as ? *as : std::string());
                    if (as) {
                        asAggr.Add(delta, *as);
                    }
                } else if (!it->second.empty()) {
                    asAggr.Add(delta, it->second);
                }
            }

            if (userAgent) {
                uaAggr.Add(delta, userAgent);
            }
        }

        TFactorsByNumberResult::TFactors res{
            .Bindings = TCountsChunk{
                .Counters = bindins.Finish(),
            },
            .Usages = TCountsChunk{
                .Counters = usages.Finish(),
            },
            .UniqUids = uidAggr.Finish(),
            .UniqIp = ipAggr.Finish(),
            .UniqAs = asAggr.Finish(),
            .UniqUserAgent = uaAggr.Finish(),
        };

        res.Bindings.MaybeMore = res.Bindings.Counters.LastYear >= settings.Limit;
        res.Usages.MaybeMore = res.Usages.Counters.LastYear >= settings.Limit;

        return res;
    }
}
