#include "pharmaapi.h"

#include "exceptions.h"
#include "factors_by_number_processor.h"
#include "runtime_context.h"
#include "utils.h"
#include "output/factors_by_number_result.h"
#include "output/serializer.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

namespace NPassport::NPharmaApi {
    static const TString PATH_PING = "/ping";
    static const TString PATH_HEALTHCHECK = "/healthcheck";

    TPharmaApi::TPharmaApi() {
        auto add = [this](const TString& key, auto handle) {
            Handlers_.emplace(key, handle);
            UnistatHandles_.Add(key, "requests." + key);
        };

        add(PATH_PING,
            [this](NCommon::TRequest& req) { HandlePing(req); });
        add(PATH_HEALTHCHECK,
            [this](NCommon::TRequest& req) { HandleHealthCheck(req); });
        add("/v1/factors_by_number",
            [this](NCommon::TRequest& req) { HandleFactorsByNumber(req); });
    }

    TPharmaApi::~TPharmaApi() = default;

    void TPharmaApi::Init(const NJson::TConfig& config) {
        const TString configPath = "/component";

        InitLogs(config, configPath);

        Runtime_ = std::make_unique<TRuntimeContext>(config, configPath);
    }

    static const TString CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_TYPE_JSON = "application/json; charset=utf-8";
    static const TString CONTENT_TYPE_TEXT = "text/plain";

    static void MakeResponse(NCommon::TRequest& request, HttpCodes code, const std::exception& e) {
        TLog::Debug() << (code < 500 ? "Bad request: " : "Internal error: ")
                      << code << "[" << (int)code << "]: " << e.what();

        request.SetStatus(code);
        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
        request.Write(TSerializer::SerializeError(e));
    };

    void TPharmaApi::HandleRequest(NCommon::TRequest& request) {
        TInstant start = TInstant::Now();
        request.ForceProvideRequestId();

        try {
            HandleImpl(request);
        } catch (const TAccessDeniedError& e) {
            MakeResponse(request, HTTP_UNAUTHORIZED, e);
        } catch (const TNotFoundError& e) {
            MakeResponse(request, HTTP_NOT_FOUND, e);
        } catch (const TInvalidValueError& e) {
            MakeResponse(request, HTTP_BAD_REQUEST, e);
        } catch (const THttpMethodError& e) {
            MakeResponse(request, HTTP_METHOD_NOT_ALLOWED, e);
        } catch (const TBadRequestError& e) {
            MakeResponse(request, HTTP_BAD_REQUEST, e);
        } catch (const std::exception& e) {
            MakeResponse(request, HTTP_INTERNAL_SERVER_ERROR, e);
        }

        LogAccess(request, TInstant::Now() - start);
    }

    void TPharmaApi::AddUnistat(NUnistat::TBuilder& builder) {
        Runtime_->AddUnistat(builder);
        UnistatHandles_.AddUnistat(builder);
    }

    void TPharmaApi::InitLogs(const NJson::TConfig& config, const TString& path) {
        config.InitCommonLog(path + "/logger");

        AccessLog_ = config.CreateLogger(path + "/access_log");
    }

    void TPharmaApi::HandleImpl(NCommon::TRequest& req) {
        req.ScanCgiFromBody();
        CheckDuplicatedArgs(req);

        TStringBuf path = req.GetPath();
        if (path.size() > 1) {
            path.ChopSuffix("/");
        }

        auto it = Handlers_.find(path);
        if (it == Handlers_.end()) {
            throw TNotFoundError() << "Path is unknown: " << path;
        }

        UnistatHandles_.Inc(path);
        it->second(req);
    }

    void TPharmaApi::CheckDuplicatedArgs(const NCommon::TRequest& req) {
        NCommon::TRequest::TDuplicatedArgs dups = req.GetDuplicatedArgs();
        if (!dups.empty()) {
            throw TInvalidValueError() << "Duplicated args are not allowed: '"
                                       << dups.begin()->first << "'";
        }
    }

    static const TString EMPTY_JSON = "{}\n";

    void TPharmaApi::HandlePing(NCommon::TRequest& req) {
        CheckHttpMethod(req, "GET");

        NJuggler::TStatus status = Runtime_->GetStatus();

        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);

        if (status == NJuggler::ECode::Critical) {
            TLog::Debug() << "PharmaApi: ping failed: " << status.Message();
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(TSerializer::SerializeError(status.Message()));
            return;
        }

        if (Runtime_->IsForceDown()) {
            TLog::Debug() << "PharmaApi: ping: service in force-down state";
            req.SetStatus(HTTP_SERVICE_UNAVAILABLE);
            req.Write(EMPTY_JSON);
            return;
        }

        req.SetStatus(HTTP_OK);
        req.Write(EMPTY_JSON);
    }

    void TPharmaApi::HandleHealthCheck(NCommon::TRequest& req) {
        CheckHttpMethod(req, "GET");

        req.Write(Runtime_->GetStatus());
        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_TEXT);
        req.SetStatus(HTTP_OK);
    }

    static const TString X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";

    void TPharmaApi::HandleFactorsByNumber(NCommon::TRequest& req) {
        CheckHttpMethod(req, "POST");

        if (req.GetQueryString()) {
            throw TBadRequestError() << "All query args must be sent only in POST body";
        }

        CheckAuth(req, "/role/factors_by_number/");

        TFactorsByNumberProcessor proc(*Runtime_);
        TFactorsByNumberResult result = proc.Process(req);

        req.Write(TSerializer::Serialize(result));
        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
        req.SetStatus(HTTP_OK);
    }

    static const TString DELIM = " ";
    static const TString DASH = "-";

    void TPharmaApi::LogAccess(const NCommon::TRequest& req, TDuration spent) {
        TStringBuf path = req.GetPath();
        if (path.size() > 1) {
            path.ChopSuffix("/");
        }

        if (path == PATH_PING || path == PATH_HEALTHCHECK) {
            return;
        }

        std::vector<TString> args;
        req.ArgNames(args);

        TString cgiParams;
        cgiParams.reserve(25 * args.size());
        for (const TString& arg : args) {
            if (!cgiParams.empty()) {
                cgiParams.push_back('&');
            }

            cgiParams.append(arg).push_back('=');
            const TString& val = req.GetArg(arg);

            if (arg == "phone_number") {
                cgiParams.append(TUtils::MaskPhoneNumber(TString(val)));
            } else {
                cgiParams.append(val);
            }
        }

        NUtils::EscapeUnprintable(cgiParams);
        if (cgiParams.empty()) {
            cgiParams = DASH;
        }

        AccessLog_->Error()
            << req.GetRemoteAddr() << DELIM
            << req.GetConsumerFormattedName() << DELIM
            << Prec(spent.MicroSeconds() / 1000.0, PREC_POINT_DIGITS, 2) << "ms" << DELIM
            << (int)req.GetStatusCode() << DELIM
            << path << DELIM
            << cgiParams;
    }

    void TPharmaApi::CheckAuth(NCommon::TRequest& req, TStringBuf expectedRole) const {
        const ui32 tvmId = Runtime_->CheckServiceTicket(req.GetHeader(X_YA_SERVICE_TICKET), expectedRole);

        req.SetConsumerFormattedName(NUtils::CreateStr("tvmid=", tvmId));
    }

    void TPharmaApi::CheckHttpMethod(const NCommon::TRequest& req, TStringBuf expectedMethod) {
        const TStringBuf actualMethod = req.GetRequestMethod();

        if (actualMethod != expectedMethod) {
            throw THttpMethodError() << "only " << expectedMethod
                                     << " allowed; got: " << actualMethod;
        }
    }
}
