#include "runtime_context.h"

#include "exceptions.h"
#include "utils.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/geobase/geobase.h>
#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/stream/file.h>
#include <util/string/join.h>
#include <util/system/fs.h>

namespace NPassport::NPharmaApi {
    TRuntimeContext::TRuntimeContext(const NJson::TConfig& config, const TString& path)
        : DbCtx_(NDbPool::TDbPoolCtx::Create())
    {
        InitTvm(config, path);
        InitDb(config, path);
        InitGeobase(config, path);
        InitFactorSettings(config, path);
        InitMisc(config, path);
        InitTestingNumbers(config, path);
    }

    TRuntimeContext::~TRuntimeContext() = default;

    void TRuntimeContext::AddUnistat(NUnistat::TBuilder& builder) const {
        DbCtx_->AddUnistat(builder);
    }

    NJuggler::TStatus TRuntimeContext::GetStatus() const {
        NJuggler::TStatus status;
        status.Update(Tvm_->GetStatus());

        TString dbMsg;
        if (!Db_->IsOk(&dbMsg)) {
            status.Update(NJuggler::ECode::Critical, status.Message(), ".", dbMsg);
        }

        return status;
    }

    ui32 TRuntimeContext::CheckServiceTicket(TStringBuf ticket, TStringBuf expectedRole) const {
        if (ticket.empty()) {
            throw TAccessDeniedError() << "Missing ServiceTicket";
        }

        NTvmAuth::TCheckedServiceTicket st = Tvm_->CheckServiceTicket(ticket);
        if (!st) {
            throw TAccessDeniedError()
                << "ServiceTicket is invalid: " << NTvmAuth::StatusToString(st.GetStatus())
                << (st.GetStatus() == NTvmAuth::ETicketStatus::InvalidDst
                        ? NUtils::CreateStr(" (expected dst in ticket: ", SelfTvmId_, ")")
                        : "")
                << "; body: '" << NTvmAuth::NUtils::RemoveTicketSignature(ticket) << "'";
        }

        if (!Tvm_->GetRoles()->CheckServiceRole(st, expectedRole)) {
            throw TAccessDeniedError()
                << "ServiceTicket is valid but src is not allowed: " << st.GetSrc();
        }

        return st.GetSrc();
    }

    bool TRuntimeContext::IsForceDown() const {
        return NFs::Exists(ForceDownFile_);
    }

    NDbPool::TDbPool& TRuntimeContext::GetDb() const {
        return *Db_;
    }

    const NGeobase::IGeobase& TRuntimeContext::GetGeobase() const {
        return *Geobase_;
    }

    const TFactorsSettings& TRuntimeContext::GetFactorsSettings() const {
        return FactorsSettings_;
    }

    const TTestingNumbers& TRuntimeContext::GetTestingNumbers() const {
        return TestingNumbers_;
    }

    void TRuntimeContext::InitTvm(const NJson::TConfig& config, const TString& path) {
        SelfTvmId_ = config.As<ui32>(path + "/tvm/self_tvm_id");

        NTvmAuth::NTvmApi::TClientSettings settings{
            .DiskCacheDir = config.As<TString>(path + "/tvm/disk_cache"),
            .SelfTvmId = SelfTvmId_,
            .Secret = (TStringBuf)NUtils::ReadFile(config.As<TString>(path + "/tvm/self_secret_file")),
            .CheckServiceTickets = true,
            .FetchRolesForIdmSystemSlug = config.As<TString>(path + "/tvm/idm_system_slug"),
            .ShouldCheckSrc = false,
        };

        Tvm_ = std::make_unique<NTvmAuth::TTvmClient>(settings, NTvmLogger::TLogger::Create());

        TLog::Info() << "TVM inited";
    }

    void TRuntimeContext::InitDb(const NJson::TConfig& config, const TString& path) {
        DbCtx_->InitLogger(config.As<TString>(path + "/dbpool_log"));

        Db_ = DbCtx_->CreateDb(config, path + "/db");
        Db_->TryPing();

        TLog::Info() << "DB connection is OK";
    }

    void TRuntimeContext::InitGeobase(const NJson::TConfig& config, const TString& path) {
        const TString filename = config.As<TString>(path + "/geobase/file");
        Geobase_ = std::make_unique<NGeobase::TGeobase>(filename);
    }

    void TRuntimeContext::InitFactorSettings(const NJson::TConfig& config, const TString& path) {
        FactorsSettings_.Limit = config.As<size_t>(path + "/factor_settings/limit");
    }

    void TRuntimeContext::InitMisc(const NJson::TConfig& config, const TString& path) {
        ForceDownFile_ = config.As<TString>(path + "/force_down_file");
    }

    void TRuntimeContext::InitTestingNumbers(const NJson::TConfig& config, const TString& path) {
        if (!config.Contains(path + "/testing_numbers_file")) {
            return;
        }

        THashSet<TString> set;
        {
            TFileInput file(config.As<TString>(path + "/testing_numbers_file"));

            TString line;
            while (file.ReadLine(line)) {
                try {
                    set.insert(TUtils::SanitizeNumber(line));
                } catch (const std::exception& e) {
                    ythrow yexception()
                        << "Failed to sanitize phone number from /testing_numbers_file: "
                        << e.what();
                }
            }
        }

        TestingNumbers_ = std::move(set);
    }
}
