#include <passport/infra/daemons/pharmaapi/src/exceptions.h>
#include <passport/infra/daemons/pharmaapi/src/factors_by_number_processor.h>
#include <passport/infra/daemons/pharmaapi/src/utils.h>
#include <passport/infra/daemons/pharmaapi/src/output/serializer.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/geobase/geobase.h>
#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/string/subst.h>

using namespace NPassport;
using namespace NPassport::NPharmaApi;

Y_UNIT_TEST_SUITE(FactorsByNumberProcessor) {
    static NDbPool::TTable CreateTable(time_t now) {
        NDbPool::TTable res;

        auto row = [&](time_t ts, ui64 uid, TString ip, TString ua, ui16 type) {
            NDbPool::TRow r;

            r.emplace_back(ToString(TUtils::InvertTs(now - ts)));
            if (uid) {
                r.emplace_back(ToString(uid));
            } else {
                r.emplace_back(NDbPool::TValue("", true));
            }
            r.emplace_back(std::move(ip));
            r.emplace_back(std::move(ua));
            r.emplace_back(ToString(type));

            return r;
        };

        res.push_back(row(3000, 0, "", "some ua #1", 1));
        res.push_back(row(3001, 0, "ip #1", "some ua #2", 1));
        res.push_back(row(40000, 0, "ip #2", "some ua #1", 1));
        res.push_back(row(40001, 0, "ip #1", "some ua #3", 1));
        res.push_back(row(100000, 0, "ip #1", "some ua #4", 1));

        res.push_back(row(2000, 143, "", "some ua #1", 2));
        res.push_back(row(2001, 143, "ip #4", "some ua #5", 2));
        res.push_back(row(30000, 145, "ip #2", "some ua #1", 2));
        res.push_back(row(30001, 143, "ip #3", "some ua #6", 2));
        res.push_back(row(110000, 143, "ip #1", "some ua #7", 2));
        res.push_back(row(110001, 144, "ip #1", "some ua #2", 2));
        res.push_back(row(110001000000, 149, "ip #1", "some ua #2", 2));

        return res;
    }

    class TTestGeobase: public NPassport::NGeobase::IGeobase {
    public:
        TTestGeobase() {
            Data = {
                {"ip #1", "as #1"},
                {"ip #2", "as #1"},
                {"ip #3", "as #3"},
            };
        }

        std::optional<ui32> GetRegionId(const std::string&) const override {
            return {};
        }

        std::optional<std::string> GetAs(const std::string& ip) const override {
            auto it = Data.find(ip);
            if (it == Data.end()) {
                return {};
            }

            return it->second;
        }

        std::map<std::string, std::string> Data;
    };

    Y_UNIT_TEST(process) {
        const time_t now = time(nullptr);

        TTestGeobase geobase;
        NTest::TRequest request;

        UNIT_ASSERT_VALUES_EQUAL(
            TFactorsByNumberProcessor::CalculateFactors({}, {}, geobase, now),
            TFactorsByNumberResult::TFactors());

        UNIT_ASSERT_VALUES_EQUAL(
            TFactorsByNumberProcessor::CalculateFactors(CreateTable(now), {.Limit = 6}, geobase, now),
            TFactorsByNumberResult::TFactors({
                .Bindings = {
                    .Counters = {
                        .LastHour = 2,
                        .LastDay = 4,
                        .LastMonth = 6,
                        .LastYear = 6,
                    },
                    .MaybeMore = true,
                },
                .Usages = {
                    .Counters = {
                        .LastHour = 2,
                        .LastDay = 4,
                        .LastMonth = 5,
                        .LastYear = 5,
                    },
                },
                .UniqUids = {
                    .LastHour = 1,
                    .LastDay = 2,
                    .LastMonth = 3,
                    .LastYear = 3,
                },
                .UniqIp = {
                    .LastHour = 2,
                    .LastDay = 4,
                    .LastMonth = 4,
                    .LastYear = 4,
                },
                .UniqAs = {
                    .LastHour = 1,
                    .LastDay = 2,
                    .LastMonth = 2,
                    .LastYear = 2,
                },
                .UniqUserAgent = {
                    .LastHour = 3,
                    .LastDay = 5,
                    .LastMonth = 7,
                    .LastYear = 7,
                },
            }));
    }
}

template <>
void Out<TFactorsByNumberResult::TFactors>(IOutputStream& o, const TFactorsByNumberResult::TFactors& value) {
    o << NPassport::NPharmaApi::TSerializer::Serialize(TFactorsByNumberResult{.Factors = value}) << Endl;
}
