package xiva

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat/unistatresty"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

const (
	backoffInitInternal   = 500 * time.Millisecond
	backoffMaxInterval    = 2000 * time.Millisecond
	backoffMaxElapsedTime = 10 * time.Second
)

type Config struct {
	URL     string         `json:"url"`
	Timeout utils.Duration `json:"timeout"`
	Retries uint64         `json:"retries"`
	Service string         `json:"service"`
}

type Client struct {
	cfg     Config
	http    *resty.Client
	tvm     tvm.Client
	unistat *unistatresty.Stats
}

func NewClient(cfg Config, tvmclient tvm.Client) (*Client, error) {
	httpc := resty.New().
		SetBaseURL(cfg.URL).
		SetTimeout(cfg.Timeout.Duration).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	res := &Client{
		cfg:  cfg,
		http: httpc,
		tvm:  tvmclient,
		unistat: unistatresty.PrepareStats(unistatresty.Config{
			Client:          httpc,
			UnistatPrefix:   "xiva",
			MaxResponseTime: 10 * cfg.Timeout.Duration,
		}),
	}

	return res, nil
}

func (x *Client) addTvm(ctx context.Context, req *resty.Request) error {
	ticket, err := x.tvm.GetServiceTicketForAlias(ctx, "xiva")
	if err != nil {
		return xerrors.Errorf("failed to get service ticket: %v", err)
	}

	req.SetHeader("X-Ya-Service-Ticket", ticket)
	return nil
}

func (x *Client) withRetries(req *resty.Request, url string, handle func(*resty.Response) error) error {
	op := func() error {
		resp, err := req.Post(url)
		if err != nil {
			return &errs.TemporaryError{Message: fmt.Sprintf("Failed to perform request to XIVA: %s", err)}
		}

		if resp.StatusCode() != http.StatusOK {
			errMsg := fmt.Sprintf("Bad response code: [%d] %s", resp.StatusCode(), resp.String())
			if resp.StatusCode() >= 400 && resp.StatusCode() < 500 && resp.StatusCode() != http.StatusTooManyRequests {
				return backoff.Permanent(&errs.BackendError{
					Message: errMsg,
				})
			}
			return &errs.TemporaryError{Message: errMsg}
		}

		return handle(resp)
	}

	notify := func(err error, delay time.Duration) {
		ctxlog.Warnf(req.Context(), logger.Log(),
			"XIVA: new retry: sleep for %s: %v", delay, err)
	}

	backoffPolicy := x.newBackoff(req.Context())

	return backoff.RetryNotify(op, backoffPolicy, notify)
}

func (x *Client) newBackoff(ctx context.Context) backoff.BackOff {
	if x.cfg.Retries <= 1 {
		return &backoff.StopBackOff{}
	}
	exp := backoff.NewExponentialBackOff()
	exp.InitialInterval = backoffInitInternal
	exp.MaxInterval = backoffMaxInterval
	exp.MaxElapsedTime = backoffMaxElapsedTime
	// Reset() must have been called after re-setting fields of ExponentialBackoff.
	// https://github.com/cenkalti/backoff/issues/69#issuecomment-448923118
	exp.Reset()

	// we doing "c.retries - 1", because of "backoff.WithMaxRetries(backoff, 2)" mean repeat operation 3 times.
	return backoff.WithContext(backoff.WithMaxRetries(exp, x.cfg.Retries-1), ctx)
}
