package pushsubscription

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ErrorResponse struct {
	Status string   `json:"status"`
	Error  []string `json:"errors"`
}

func (t *PushSubscription) handleErrorMiddleware() echo.MiddlewareFunc {
	// This is API for external consumers, so we cannot show detailed error message

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			err := next(ctx)

			switch err := err.(type) {
			case nil:
				return nil

			case *echo.HTTPError:
				t.unistat.errInvalidParams.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "EchoError: %s", err)
				return sendErrorResponseImpl(ctx, err.Code, fmt.Sprintf("%v", err.Message))

			case *errs.InvalidParamError:
				t.unistat.errInvalidParams.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "InvalidParam: %s", err)
				return sendErrorResponseImpl(ctx, http.StatusOK, fmt.Sprintf("%s.%s", err.Key, err.Type))

			case *errs.BackendError:
				t.unistat.errBackend.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "BackendError: %s", err)
				return sendErrorResponseImpl(ctx, http.StatusOK, "backend.failed")

			case *errs.TemporaryError:
				t.unistat.errTmp.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "TemporaryError: %s", err)
				return sendErrorResponseImpl(ctx, http.StatusInternalServerError, "internal error")
			}

			t.unistat.errUnknown.Inc()
			ctxlog.Errorf(ctx.Request().Context(), logger.Log(), "Unknown error: %s", err)
			return sendErrorResponseImpl(ctx, http.StatusInternalServerError, "internal error")
		}
	}
}

func sendErrorResponseImpl(ctx echo.Context, code int, err string) error {
	return ctx.JSON(code, ErrorResponse{
		Status: "error",
		Error:  []string{err},
	})
}
