package pushsubscription

import (
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/model/bb"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/model/xiva"
	"a.yandex-team.ru/passport/infra/libs/go/tskv"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type PushSubscription struct {
	tvm          tvm.Client
	auth         model.AuthProviders
	subscriber   model.PushProviders
	cfg          Config
	accessLog    log.Logger
	tskvPassport *tskv.FileLogger
	tskvPush     *tskv.FileLogger
	unistat      stats
}

type CommonConfig struct {
	ForceDownFile       string        `json:"force_down_file"`
	AccessLog           logger.Config `json:"access_log"`
	PassportStatboxLog  string        `json:"passport_statbox_log"`
	PushSubscriptionLog string        `json:"push_subscription_log"`
}

type Config struct {
	Common CommonConfig       `json:"common"`
	Tvm    httpdtvm.TvmConfig `json:"tvm"`
	Bb     bb.Config          `json:"blackbox"`
	Xiva   xiva.Config        `json:"xiva"`
}

type stats struct {
	errInvalidParams *unistat.SignalDiff
	errTmp           *unistat.SignalDiff
	errBackend       *unistat.SignalDiff
	errUnknown       *unistat.SignalDiff
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	bbClient, err := bb.NewClient(cfg.Bb, tvmClient, cfg.Tvm.Destinations["blackbox"])
	if err != nil {
		return nil, err
	}

	xivaClient, err := xiva.NewClient(cfg.Xiva, tvmClient)
	if err != nil {
		return nil, err
	}

	accessLog, err := logger.CreateLog(cfg.Common.AccessLog)
	if err != nil {
		return nil, err
	}

	tskvPassport, err := tskv.NewFileLogger(tskv.FileConfig{
		Filename: cfg.Common.PassportStatboxLog,
		Format:   "passport-log",
	})
	if err != nil {
		return nil, err
	}

	tskvPush, err := tskv.NewFileLogger(tskv.FileConfig{
		Filename: cfg.Common.PushSubscriptionLog,
		Format:   "push-subscription-log",
	})
	if err != nil {
		return nil, err
	}

	res := &PushSubscription{
		tvm:          tvmClient,
		auth:         bbClient,
		subscriber:   xivaClient,
		cfg:          cfg,
		accessLog:    accessLog,
		tskvPassport: tskvPassport,
		tskvPush:     tskvPush,
		unistat: stats{
			errInvalidParams: unistat.DefaultChunk.CreateSignalDiff("errors.requests.invalid_params"),
			errTmp:           unistat.DefaultChunk.CreateSignalDiff("errors.requests.tmp_err"),
			errBackend:       unistat.DefaultChunk.CreateSignalDiff("errors.requests.backend"),
			errUnknown:       unistat.DefaultChunk.CreateSignalDiff("errors.requests.unknown"),
		},
	}
	return res, nil
}

func (t *PushSubscription) AddHandlers(e *echo.Echo) {
	e.Use(
		handleAccessLogMiddleware(t.accessLog),
		t.handleErrorMiddleware(),
	)

	e.GET(
		"/ping",
		t.HandlePing(),
	)

	e.GET(
		"/healthcheck",
		t.HandleHealthCheck(),
	)

	e.POST(
		"/1/bundle/push/subscribe/",
		t.HandleSubscribe(),
		t.handleClientIP(),
	)
	e.POST(
		"/1/bundle/push/unsubscribe/",
		t.HandleUnsubscribe(),
		t.handleClientIP(),
	)
}

func responseStatusOk() interface{} {
	type t struct {
		Status string `json:"status"`
	}

	return &t{Status: "ok"}
}

func (t *PushSubscription) newTskvPasspLineWithDefaultFields(ctx echo.Context) *tskv.LineToFile {
	now := time.Now()

	line := t.tskvPassport.NewLine()
	line.
		// time
		AddValue("timestamp", now.Local().Format("2006-01-02T15:04:05"), false).
		AddValue("timezone", now.Local().Format("-0700"), false).
		AddValue("unixtime", strconv.FormatInt(now.Unix(), 10), false).
		// from context
		AddValue("ip", ContextClientIP(ctx.Request().Context()).String(), false).
		AddValue("request_id", middlewares.ContextReqID(ctx.Request().Context()), false).
		AddValue("user_agent", ctx.Request().Header.Get("User-Agent"), true).
		// const
		AddValue("service_name", t.cfg.Xiva.Service, false).
		AddValue("mode", "push", false).
		AddValue("py", "1", false)

	return line
}
