package pushsubscription

import (
	"crypto/md5"
	"encoding/hex"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/push_subscription/internal/reqs"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
)

const (
	pushAppNamePostfixAndroid = ".passport"
)

var supportedPlatforms = map[string]string{
	"android": "gcm",
	"apns":    "apns",
	"apple":   "apns",
	"gcm":     "gcm",
	"ipad":    "apns",
	"iphone":  "apns",
}

func (t *PushSubscription) HandleSubscribe() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		req, err := parseSubscribeRequest(ctx)
		if err != nil {
			return err
		}

		err = subscribeImpl(
			ctx,
			req,
			t.auth,
			t.subscriber,
			func(uid string, uuid string) {
				tskvPassportLine := t.newTskvPasspLineWithDefaultFields(ctx)
				tskvPassportLine.
					AddValue("action", "subscribed", false).
					AddValue("app_id", req.AppID, true).
					AddValue("app_platform", req.AppPlatform, true).
					AddValue("uid", uid, false).
					AddValue("uuid", uuid, false)
				tskvPassportLine.Write()
			},
			func(uid string, uuid string) {
				tskvPushLine := t.tskvPush.NewLine()
				tskvPushLine.
					AddValue("app_id", req.AppID, true).
					AddValue("device_id", req.DeviceID, true).
					AddValue("request_id", middlewares.ContextReqID(ctx.Request().Context()), false).
					AddValue("uid", uid, false).
					AddValue("unixtime", strconv.FormatInt(time.Now().Unix(), 10), false).
					AddValue("is_push_token_upgrade_required", fmt.Sprintf("%t", req.IsPushTokenUpgradeRequired), false)
				tskvPushLine.Write()
			},
		)
		if err != nil {
			return err
		}

		return ctx.JSON(http.StatusOK, responseStatusOk())
	}
}

func parseSubscribeRequest(ctx echo.Context) (*reqs.SubscribeRequest, error) {
	err := ctx.Request().ParseForm()
	if err != nil {
		return nil, err
	}

	query := ctx.Request().Form

	appID, err := getRequiredStringParam(query, "app_id")
	if err != nil {
		return nil, err
	}
	appPlatform, err := getRequiredStringParam(query, "app_platform")
	if err != nil {
		return nil, err
	}
	deviceid, err := getRequiredStringParam(query, "deviceid")
	if err != nil {
		return nil, err
	}
	deviceToken, err := getRequiredStringParam(query, "device_token")
	if err != nil {
		return nil, err
	}
	isPushTokenUpgradeRequired, err := getDefaultBoolParam(query, "is_push_token_upgrade_required", true)
	if err != nil {
		return nil, err
	}

	return &reqs.SubscribeRequest{
		AppID:                      appID,
		AppPlatform:                appPlatform,
		DeviceID:                   deviceid,
		DeviceToken:                deviceToken,
		AmVersion:                  query.Get("am_version"),
		AppVersion:                 query.Get("app_version"),
		Client:                     query.Get("client"),
		Filter:                     query.Get("filter"),
		IsPushTokenUpgradeRequired: isPushTokenUpgradeRequired,
	}, nil
}

func translatePlatform(platform string) (string, error) {
	platform = strings.ToLower(platform)

	for key, value := range supportedPlatforms {
		if strings.HasPrefix(platform, key) {
			return value, nil
		}
	}

	return "", &errs.InvalidParamError{
		Key:     "push_api",
		Type:    "app_platform_unsupported",
		Message: fmt.Sprintf("got illegal platform: '%s'", platform),
	}
}

func getAppName(appPlatform, appID string) string {
	if appPlatform == "gcm" && !strings.HasSuffix(appID, pushAppNamePostfixAndroid) {
		return appID + pushAppNamePostfixAndroid
	}

	return appID
}

func createUUID(deviceid, appID string) string {
	hash := md5.Sum([]byte(deviceid + appID))
	return hex.EncodeToString(hash[:])
}

func subscribeImpl(
	ctx echo.Context,
	req *reqs.SubscribeRequest,
	auth model.OAuthProvider,
	subscriber model.SubscribeProvider,
	logPasspFunc func(uid string, uuid string),
	logPushFunc func(uid string, uuid string),
) error {
	platform, err := translatePlatform(req.AppPlatform)
	if err != nil {
		return err
	}

	authResponse, err := checkAuthorization(ctx, auth)
	if err != nil {
		return err
	}

	uid := fmt.Sprintf("%d", authResponse.UID)
	uuid := createUUID(req.DeviceID, req.AppID)

	if !req.IsPushTokenUpgradeRequired {
		logPushFunc(uid, uuid)
		return nil
	}

	_, err = subscriber.Subscribe(ctx.Request().Context(), &model.SubscribeRequest{
		AppName: getAppName(platform, req.AppID),
		// Client:   req.Client, // TODO: python doesn't send it
		DeviceID: req.DeviceID,
		Extra: model.SubscribeExtra{
			AmVersion:  req.AmVersion,
			LoginID:    authResponse.LoginID,
			AppVersion: req.AppVersion,
		},
		// Filter:    req.Filter, // TODO: python doesn't send it
		Platform:  platform,
		PushToken: req.DeviceToken,
		UUID:      uuid,
		User:      uid,
	})
	if err != nil {
		return err
	}

	logPasspFunc(uid, uuid)
	logPushFunc(uid, uuid)

	return err
}
