package sesskill

import (
	"context"
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/gocql/gocql"
	"github.com/labstack/echo/v4"
)

type SessionAliveRequest struct {
	AuthID string
	UIDs   []string
}

type SessionAliveResponseRow struct {
	UID string
	TS  uint64
}

type SessionAliveResponse []*SessionAliveResponseRow

func (r SessionAliveResponse) String() string {
	var result strings.Builder
	for _, row := range r {
		result.WriteString(fmt.Sprintf("%s:%d\n", row.UID, row.TS))
	}

	return result.String()
}

type SessionAliveInvalidParamError struct {
	Code    int
	Key     string
	Message string
}

func (e *SessionAliveInvalidParamError) Error() string {
	return e.Message
}

type SessionAliveCassandraError struct {
	Code    int
	Message string
}

func (e *SessionAliveCassandraError) Error() string {
	return e.Message
}

func ParseSessionAliveRequest(c echo.Context) (*SessionAliveRequest, error) {
	query := c.Request().URL.Query()

	//
	// обработка параметров повторяет поведение scala legacy api
	// (местами это поведение нелогично, но оставлено в целях совместимости)
	//

	if !query.Has("authid") {
		return nil, &SessionAliveInvalidParamError{
			Code:    http.StatusNotFound,
			Key:     "authid",
			Message: "missing 'authid' param",
		}
	}

	authid := query.Get("authid")
	if len(authid) == 0 {
		return nil, &SessionAliveInvalidParamError{
			Code:    http.StatusInternalServerError,
			Key:     "authid",
			Message: "empty authid param",
		}
	}

	if !query.Has("uids") {
		return nil, &SessionAliveInvalidParamError{
			Code:    http.StatusNotFound,
			Key:     "uids",
			Message: "missing 'uids' param",
		}
	}

	uids := query.Get("uids")
	if len(uids) == 0 {
		return nil, &SessionAliveInvalidParamError{
			Code:    http.StatusBadRequest,
			Key:     "uids",
			Message: "empty uids param",
		}
	}

	result := &SessionAliveRequest{
		AuthID: authid,
		UIDs:   make([]string, 0),
	}

	list := strings.Split(uids, ",")
	for _, item := range list {
		if len(item) == 0 {
			continue
		}

		uid, err := strconv.ParseInt(item, 10, 64)
		if err != nil {
			return nil, &SessionAliveInvalidParamError{
				Code:    http.StatusBadRequest,
				Key:     "uids",
				Message: fmt.Sprintf("invalid uid: '%s'", item),
			}
		}

		result.UIDs = append(result.UIDs, strconv.FormatInt(uid, 10))
	}

	return result, nil
}

func (t *SessKill) HandleSessionAlive() echo.HandlerFunc {
	return func(c echo.Context) error {
		req, err := ParseSessionAliveRequest(c)
		if err != nil {
			return t.sendErrorResponse(c, err)
		}

		if len(req.UIDs) == 0 {
			return c.String(http.StatusOK, "")
		}

		result, err := t.GetSessionAlive(c.Request().Context(), req)
		if err != nil {
			return t.sendErrorResponse(c, &SessionAliveCassandraError{
				Code:    http.StatusInternalServerError,
				Message: err.Error(),
			})
		}

		if len(result) != 0 {
			c.Response().Header().Set(echo.HeaderContentType, "text/plain; charset=UTF-8")
		}

		return c.String(http.StatusOK, result.String())
	}
}

func (t *SessKill) GetSessionAlive(ctx context.Context, req *SessionAliveRequest) (SessionAliveResponse, error) {
	// нельзя использовать Query(...).WithContext(ctx)
	// т.к. отмена контекста приводит к отмене всех запросов в сессии
	// (похоже на проблему https://github.com/gocql/gocql/issues/1341)
	// в качестве workaround используется глобальный таймаут кластера
	scanner := t.session.Query(
		"SELECT uid, ts FROM ses_kills where authid=? and uid in ?",
		req.AuthID,
		req.UIDs,
	).Consistency(gocql.LocalOne).Iter().Scanner()

	var err error
	result := make(SessionAliveResponse, 0)
	for scanner.Next() {
		row := &SessionAliveResponseRow{}
		err = scanner.Scan(&row.UID, &row.TS)
		if err != nil {
			_ = scanner.Err() // release scanner
			return nil, err
		}

		result = append(result, row)
	}

	if err := scanner.Err(); err != nil {
		return nil, err
	}

	return result, nil
}
