package sesskill

import (
	"fmt"
	"net/http"
	"time"

	"github.com/gocql/gocql"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type stats struct {
	invalidParams *unistat.SignalDiff
	cassandra     *unistat.SignalDiff
	unknown       *unistat.SignalDiff
}

type SessKill struct {
	tvm       tvm.Client
	unistat   stats
	accessLog log.Logger
	cassandra *gocql.ClusterConfig
	session   *gocql.Session
	debug     bool
}

type CassandraConfig struct {
	Keyspace string   `json:"keyspace"`
	Hosts    []string `json:"hosts"`
	Timeout  uint64   `json:"timeout"`
}

type CommonConfig struct {
	Debug     bool          `json:"debug"`
	AccessLog logger.Config `json:"access_log"`
}

type Config struct {
	Tvm       httpdtvm.TvmConfig `json:"tvm"`
	Common    CommonConfig       `json:"common"`
	Cassandra CassandraConfig    `json:"cassandra"`
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	debug := cfg.Common.Debug
	accessLog, err := logger.CreateLog(cfg.Common.AccessLog)
	if err != nil {
		return nil, err
	}

	cassandra := gocql.NewCluster(cfg.Cassandra.Hosts...)
	cassandra.Keyspace = cfg.Cassandra.Keyspace
	cassandra.PoolConfig.HostSelectionPolicy = gocql.RoundRobinHostPolicy()
	cassandra.Compressor = gocql.SnappyCompressor{}

	// грубый расчет в PASSP-35847
	cassandra.Timeout = time.Duration(cfg.Cassandra.Timeout) * time.Millisecond
	cassandra.NumConns = 16
	cassandra.SocketKeepalive = 20 * time.Second

	session, err := cassandra.CreateSession()
	if err != nil {
		return nil, err
	}

	res := &SessKill{
		tvm: tvmClient,
		unistat: stats{
			invalidParams: unistat.DefaultChunk.CreateSignalDiff("errors.requests.invalid_params"),
			cassandra:     unistat.DefaultChunk.CreateSignalDiff("errors.requests.cassandra"),
			unknown:       unistat.DefaultChunk.CreateSignalDiff("errors.requests.unknown"),
		},
		accessLog: accessLog,
		cassandra: cassandra,
		session:   session,
		debug:     debug,
	}
	return res, nil
}

func (t *SessKill) AddHandlers(e *echo.Echo) {
	e.Pre(t.middlewareAccessLog())

	e.GET(
		"/ping",
		t.HandlePing(),
	)

	e.GET(
		"/1/profile/",
		t.HandleEmpty(),
	)

	e.GET(
		"/1/phones/stats/",
		t.HandleEmpty(),
	)

	e.GET(
		"/1/session/alive/",
		t.HandleSessionAlive(),
	)
}

func (t *SessKill) HandlePing() echo.HandlerFunc {
	return func(c echo.Context) error {
		if t.debug {
			logger.Log().Debugf("Ping: service is up")
		}
		return c.String(http.StatusOK, "pong")
	}
}

func (t *SessKill) HandleEmpty() echo.HandlerFunc {
	return func(c echo.Context) error {
		return c.String(http.StatusOK, "")
	}
}

func (t *SessKill) sendErrorResponse(c echo.Context, err error) error {
	if err == nil {
		err = xerrors.New("some error occurred but original message was lost")
	}

	switch err := err.(type) {
	case *SessionAliveInvalidParamError:
		t.unistat.invalidParams.Inc()
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", err)
		return c.String(err.Code, "")
	case *SessionAliveCassandraError:
		t.unistat.cassandra.Inc()
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", err)
		return c.String(err.Code, "")
	}

	t.unistat.unknown.Inc()
	ctxlog.Errorf(c.Request().Context(), logger.Log(), "Unexpected error: %s", err)
	return c.String(http.StatusInternalServerError, "")
}

func (t *SessKill) middlewareAccessLog() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				if t.debug || c.Response().Status != http.StatusOK {
					startTime := middlewares.ContextStartInstant(c.Request().Context())

					st := tvm.ContextServiceTicket(c.Request().Context())
					ut := tvm.ContextUserTicket(c.Request().Context())

					srcID := "-"
					if st != nil {
						srcID = fmt.Sprintf("%d", st.SrcID)
					}

					defaultUID := "-"
					if ut != nil {
						defaultUID = fmt.Sprintf("%d", ut.DefaultUID)
					}

					query := c.Request().URL.RawQuery
					if query == "" {
						query = "-"
					}

					t.accessLog.Debugf(
						"%s\t%s\t%d\t%.2fms\ttvmid=%s\tuser=%s\t%s\t%s\t%s",
						middlewares.ContextReqID(c.Request().Context()),
						c.RealIP(),
						c.Response().Status,
						float64(time.Since(startTime).Microseconds())/1000,
						srcID,
						defaultUID,
						c.Request().Method,
						c.Request().URL.Path,
						query,
					)
				}
			})

			return next(c)
		}
	}
}
