#include "reg_completion_recommended.h"

#include <passport/infra/daemons/sezamapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/yp_cookie_parser/yp_cookie_parser.h>

#include <library/cpp/blackbox2/blackbox2.h>

#include <util/generic/yexception.h>

namespace NPassport::NSezamApi {
    static const TString COOKIE_LRCS_ = "lrcs";
    static const TString EMPTY = R"({})";

    static const TString BB_QUERY_PATH = "/blackbox";
    static const TString METHOD_POST = "POST";

    static const TString YP_COOKIE_DOMAIN = "yandex.ru";

    TString TRegCompletionArgs::ToQuery() const {
        const TString query = NBlackbox2::SessionIDRequest(Sessionid, Domain, Userip);
        if (YPCookie) {
            return NUtils::CreateStr(query, "&get_reg_completion_recommendation_with_yp=", YPCookie.value());
        }
        return query;
    }

    NDbPool::TQuery TRegCompletionRecommended::BuildBbQuery(const TRegCompletionArgs& args) {
        NDbPool::TQuery query(BB_QUERY_PATH);
        query.SetHttpMethod(METHOD_POST);
        query.SetHttpBody(
            NUtils::CreateStr(
                args.ToQuery(),
                "&sessguard=",
                NUtils::Urlencode(args.Sessguard)));
        return query;
    }

    THolder<NBlackbox2::TSessionResp> TRegCompletionRecommended::GetBbSessionIdResponse(NDbPool::TDbPool& db,
                                                                                        const TRegCompletionArgs& args) {
        try {
            const TString body = NDbPool::NUtils::GetHttpBody(
                db,
                BuildBbQuery(args),
                "Blackbox error");
            THolder<NBlackbox2::TSessionResp> user = NBlackbox2::SessionIDResponse(body);

            if (user->Status() != NBlackbox2::TSessionResp::Valid && user->Status() != NBlackbox2::TSessionResp::NeedReset) {
                TLog::Warning() << "Cookie is invalid: " << ToString(user->Status());
                return nullptr;
            }
            return user;
        } catch (const std::exception& e) {
            TLog::Warning() << "Error: SidsModule/registration_status/check: failed to get data from blackbox: " << e.what();
            throw TBackendException() << e.what();
        }
    }

    bool TRegCompletionRecommended::CheckRegCompletionRecommended(NDbPool::TDbPool& db,
                                                                  const TRegCompletionArgs& args) {
        THolder<NBlackbox2::TSessionResp> user = GetBbSessionIdResponse(db, args);
        if (!user) {
            return false;
        }
        NBlackbox2::TRegistrationCompletedInfo regCompletedInfo(user.Get());
        return regCompletedInfo.NeedRegCompleted();
    }

    TString TRegCompletionRecommended::GetConnectionId(NDbPool::TDbPool& db,
                                                       const TRegCompletionArgs& args) {
        THolder<NBlackbox2::TSessionResp> user = GetBbSessionIdResponse(db, args);
        if (!user) {
            return "";
        }
        NBlackbox2::TConnectionId connectionId(user.Get());
        return connectionId.ConnectionId();
    }

    TString TRegCompletionRecommended::SerializeRegCompletionRecommended(bool checkRegistrationCompleted) {
        TString body;
        NJson::TWriter wr(body);
        NJson::TObject root(wr);
        root.Add("is_completion_recommended", checkRegistrationCompleted);
        return body;
    }

    void CheckRegCompetionRecommendedBlackboxGrants(NDbPool::TDbPool& bb) {
        NDbPool::TQuery query = TRegCompletionRecommended::BuildBbQuery(
            TRegCompletionArgs{});

        TUtils::CheckBlackboxGrants(bb, std::move(query));
    }

    void TRegCompletionRecommended::SetLrcsToYpCookie(NCommon::TRequest& req, ui32 lrcsExpirationTtl, TInstant now) {
        const TString& ypCookieOldValue = req.GetCookie(COOKIE_YP_);
        TString ypCookieValue;
        NPassport::NYpCookie::TYpCookieElem lrcsCookie{ToString(now.TimeT() + lrcsExpirationTtl), COOKIE_LRCS_, ToString(now.TimeT())};

        if (!NPassport::NYpCookie::TryUpdateYpCookie(lrcsCookie, ypCookieOldValue, ypCookieValue)) {
            TLog::Warning() << "Yp cookie was invalid, do not update it: " << ypCookieOldValue;
            req.SetStatus(HttpCodes::HTTP_PARTIAL_CONTENT);
            return;
        }

        NCommon::TCookie yp(COOKIE_YP_, ypCookieValue);
        yp.SetHttpOnly(true);
        yp.SetSecure(true);
        yp.SetDomain(TUtils::PrepareYpCookieDomain(req.GetHost()));
        yp.SetSameSite(NCommon::TCookie::ESameSite::None);
        req.SetCookie(yp);
    }

    const TString& TRegCompletionRecommended::DefaultAnswer() {
        return EMPTY;
    }
}
