#include "runtime_context.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/fs.h>

namespace NPassport::NSezamApi {
    TRuntimeContext::TRuntimeContext(const NXml::TConfig& config, const TString& path)
        : DbCtx_(NDbPool::TDbPoolCtx::Create())
    {
        DbCtx_->InitLogger(config.AsString(path + "/log_dbpool"));
        ForceDownFile_ = config.AsString(path + "/force_down_file");

        InitTvm(config, path + "/tvm");

        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            TvmClient_,
            NTvmCommon::TServiceTickets::BLACKBOX_)};
        Bb_ = DbCtx_->CreateDb(config, path + "/blackbox", std::move(opts));
        Bb_->TryPing();

        TimingsCfg_.DoNotDisturbAfterClose =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_close", 7 * 24 * 60 * 60);
        TimingsCfg_.DoNotDisturbAfterCloseMin =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_close_min", 60 * 60);
        TimingsCfg_.DoNotDisturbAfterCloseMax =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_close_max", 90 * 24 * 60 * 60);
        TimingsCfg_.DoNotDisturbAfterShow =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_show", 24 * 60 * 60);
        TimingsCfg_.DoNotDisturbAfterShowMin =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_show_min", 60 * 60);
        TimingsCfg_.DoNotDisturbAfterShowMax =
            config.AsInt(path + "/suggest_accounts/do_not_disturb_after_show_max", 24 * 60 * 60);

        TimingsCfg_.IlahuTtl = config.AsInt(path + "/suggest_accounts/ilahu_ttl", 30 * 24 * 60 * 60);
        TimingsCfg_.LrcsExpirationTtl = config.AsInt(path + "/reg_completion_recommended/lrcs_ttl", 2 * 30 * 24 * 60 * 60);

        const TOrigin::EPolicy policy = PolicyFromString(
            config.AsString(path + "/origin/policy", "strict"));
        Origins_ = std::make_unique<TOrigins>();

        Origins_->YandexTld = TOrigin::CreateForYandexTld(
            policy,
            config.AsString(path + "/origin/yandex_sld"),
            DomainListFromCfg(config, path + "/origin/yandex_tld"));
        Origins_->SideDomainsDefault = TOrigin::CreateForSideDomains(
            policy,
            DomainListFromCfg(config, path + "/origin/side_domains_default"));
        Origins_->SideDomainsForAccounts = TOrigin::CreateForSideDomains(
            policy,
            DomainListFromCfg(config, path + "/origin/side_domains_accounts"));
        Origins_->SideDomainsForRegCompletionRecommended = TOrigin::CreateForSideDomains(
            policy,
            DomainListFromCfg(config, path + "/origin/side_domains_reg_completion_recommended"));

        if (!Origins_->SideDomainsDefault) {
            TLog::Info() << "There is no side domains for default handles";
        }
        if (!Origins_->SideDomainsForAccounts) {
            TLog::Info() << "There is no side domains for 'accounts' handles";
        }
        if (!Origins_->SideDomainsForRegCompletionRecommended) {
            TLog::Info() << "There is no side domains for 'registration completion recommended' handles";
        }

        InitMethods(config, path);
    }

    TRuntimeContext::~TRuntimeContext() = default;

    bool TRuntimeContext::IsForceDown() const {
        return NFs::Exists(ForceDownFile_);
    }

    void TRuntimeContext::AddUnistat(NUnistat::TBuilder& builder) const {
        DbCtx_->AddUnistat(builder);
    }

    NDbPool::TDbPool& TRuntimeContext::Bb() {
        return *Bb_;
    }

    bool TRuntimeContext::IsOk(TString& out) const {
        TString tmp;

        bool bb = Bb_->IsOk(&tmp);
        if (!bb) {
            out = tmp;
        }

        NTvmAuth::TClientStatus status = TvmClient_->GetStatus();
        if (status != NTvmAuth::TClientStatus::Ok) {
            NUtils::AppendSeparated(out, ";", status.GetLastError());
        }

        return bb && (status == NTvmAuth::TClientStatus::Ok);
    }

    void TRuntimeContext::InitTvm(const NXml::TConfig& config, const TString& path) {
        NTvmAuth::NTvmApi::TClientSettings tvmSettings;
        tvmSettings.SetSelfTvmId(config.AsNum<ui32>(path + "/self_tvmid"));
        tvmSettings.SetTvmHostPort(config.AsString(path + "/host"),
                                   config.AsNum<ui32>(path + "/port"));
        tvmSettings.SetDiskCacheDir(config.AsString(path + "/disk_cache_dir"));
        tvmSettings.EnableServiceTicketsFetchOptions(
            NUtils::ReadFile(config.AsString(path + "/secret_path")),
            {{NTvmCommon::TServiceTickets::BLACKBOX_, config.AsNum<ui32>(path + "/bb_tvmid")}});

        TvmClient_ = std::make_shared<NTvmAuth::TTvmClient>(std::move(tvmSettings), NTvmLogger::TLogger::Create());
    }

    void TRuntimeContext::InitMethods(const NXml::TConfig& config, const TString& path) {
        for (const TString& p : config.SubKeys(path + "/methods/method")) {
            ui32 id = config.AsNum<ui32>(p + "/@id");
            TString name = config.AsString(p + "/@name");

            Y_ENSURE(id != 0, "method/@id cannot be 0 by buisness logic");
            Y_ENSURE(MethodsCfg_.Mapping.emplace(id, name).second,
                     "duplicated method/@id in config: " << id);
        }

        TLog::Info() << "Got " << MethodsCfg_.Mapping.size() << " method mappings";
    }

    static const TString STRICT = "strict";
    static const TString RELAXED = "relaxed";
    TOrigin::EPolicy TRuntimeContext::PolicyFromString(const TString& s) {
        if (s == STRICT) {
            return TOrigin::EPolicy::Strict;
        }
        if (s == RELAXED) {
            return TOrigin::EPolicy::Relaxed;
        }
        ythrow yexception() << "Unexepected value of origin/policy '" << s << "'. "
                            << "Allowed only: '" << STRICT << "', '" << RELAXED << "'";
    }

    std::vector<TString> TRuntimeContext::DomainListFromCfg(const NXml::TConfig& config,
                                                            const TString& p) {
        std::vector<TString> domains = config.SubKeys(p + "/domain");

        for (TString& d : domains) {
            d = config.AsString(d);
        }

        return domains;
    }
}
