#include "sezamapi.h"

#include "runtime_context.h"
#include "suggest_processor.h"

#include <passport/infra/daemons/sezamapi/src/reg_completion_handles/reg_completion_recommended.h>
#include <passport/infra/daemons/sezamapi/src/sids/sids_accounts.h>
#include <passport/infra/daemons/sezamapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/generic/strbuf.h>

namespace NPassport::NSezamApi {
    static const TString ERROR_ = "error";

    class TBadRequest: public yexception {
    public:
        TString ToJson() const {
            TString res;

            NJson::TWriter wr(res);
            NJson::TObject root(wr);

            root.Add(ERROR_, what());

            return res;
        }
    };

    static TString BuildHandleKey(const TStringBuf method, const TStringBuf path) {
        return NUtils::CreateStr(method, ".", path);
    }

    TSezamApi::TSezamApi() {
        auto addUnistat = [this](const TString& method, const TString& path) {
            TString key = BuildHandleKey(method, path);
            UnistatHandles_.Add(key, "requests." + key);
        };
        addUnistat("OPTIONS", "");

        HandlersMethod_.emplace(
            "GET",
            [this](NCommon::TRequest& req, TStringBuf path) { ProcessGet(req, path); });
        HandlersMethod_.emplace(
            "POST",
            [this](NCommon::TRequest& req, TStringBuf path) { ProcessPost(req, path); });
        HandlersMethod_.emplace(
            "OPTIONS",
            [this](NCommon::TRequest& req, TStringBuf path) { ProcessOptions(req, path); });

        auto addHandleGet = [this, addUnistat](const TString& path, auto handle) {
            addUnistat("GET", path);
            addUnistat("OPTIONS", path);
            HandlersGet_.emplace(path, handle);
        };
        addHandleGet("/nagios",
                     [this](NCommon::TRequest& req) { ProcessScriptNagios(req); });
        addHandleGet("/ping",
                     [this](NCommon::TRequest& req) { ProcessScriptNagios(req); });
        addHandleGet("/suggested_accounts",
                     [this](NCommon::TRequest& req) { ProcessScriptSuggest(req); });
        addHandleGet("/accounts",
                     [this](NCommon::TRequest& req) { ProcessScriptAccounts(req, EOnlyValidUsers::True); });
        addHandleGet("/all_accounts",
                     [this](NCommon::TRequest& req) { ProcessScriptAccounts(req, EOnlyValidUsers::False); });
        addHandleGet("/registration_status/check",
                     [this](NCommon::TRequest& req) { ProcessScriptRegistrationStatusCheck(req); });

        auto addHandlePost = [this, addUnistat](const TString& path, auto handle) {
            addUnistat("POST", path);
            addUnistat("OPTIONS", path);
            HandlersPost_.emplace(path, handle);
        };
        addHandlePost("/suggested_accounts/close",
                      [this](NCommon::TRequest& req) { ProcessScriptCloseSuggest(req); });
        addHandlePost("/suggested_accounts/show_and_postpone",
                      [this](NCommon::TRequest& req) { ProcessScriptSuggestShowAndPostpone(req); });
        addHandlePost("/registration_status/postpone",
                      [this](NCommon::TRequest& req) { ProcessScriptRegistrationStatusPostpone(req); });
    }

    TSezamApi::~TSezamApi() = default;

    void TSezamApi::Init(const NXml::TConfig& config) {
        try {
            const TString configPath = "/config/components/component";

            config.InitCommonLog(configPath + "/logger_common");
            TLog::Info() << "SezamApi: starting service";

            Runtime_ = std::make_unique<TRuntimeContext>(config, configPath);
            AccountsSettings_ = std::make_unique<TAccountsSettings>(config, configPath + "/accounts_settings");
            CheckGrants();
        } catch (const std::exception& e) {
            TLog::Error() << "Lah server failed to start: " << e.what();
            throw;
        }
    }

    static const TString CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_TYPE_TEXT_PLAIN = "text/plain; charset=utf-8";
    static const TString CONTENT_TYPE_JSON = "application/json; charset=utf-8";
    static const TString CONTENT_TYPE_OPTIONS = "X-Content-Type-Options";
    static const TString CONTENT_TYPE_OPTIONS_BODY = "nosniff";
    void TSezamApi::HandleRequest(NCommon::TRequest& request) {
        request.ScanCgiFromBody();

        try {
            CheckDuplicatedArgs(request);
            ChooseMethod(request);
        } catch (const TBadRequest& e) {
            TLog::Debug() << "Bad request: " << e.what();

            request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
            request.SetHeader(CONTENT_TYPE_OPTIONS, CONTENT_TYPE_OPTIONS_BODY);
            request.SetStatus(HTTP_NOT_FOUND);
            request.Write(e.ToJson());
        } catch (const TBackendException&) {
            request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
            request.SetHeader(CONTENT_TYPE_OPTIONS, CONTENT_TYPE_OPTIONS_BODY);
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            request.Write("{}\n");
        } catch (const std::exception& e) {
            const TString msg = NUtils::CreateStr("Unexpected error: ", e.what());
            TLog::Error() << msg;

            request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_TEXT_PLAIN);
            request.SetHeader(CONTENT_TYPE_OPTIONS, CONTENT_TYPE_OPTIONS_BODY);
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            request.Write(msg);
        }
    }

    void TSezamApi::AddUnistat(NUnistat::TBuilder& builder) {
        Runtime_->AddUnistat(builder);
        UnistatHandles_.AddUnistat(builder);
    }

    void TSezamApi::AddUnistatExtended(const TString& path, NUnistat::TBuilder& builder) {
        if (path == "/origins/") {
            UnistatOrigins_.AddUnistat(builder);
        }
    }

    void TSezamApi::CheckDuplicatedArgs(const NCommon::TRequest& req) {
        NCommon::TRequest::TDuplicatedArgs dups = req.GetDuplicatedArgs();
        if (!dups.empty()) {
            throw TBadRequest()
                << "Duplicate args not allowed, got several values of '"
                << dups.begin()->first << "'";
        }
    }

    void TSezamApi::ChooseMethod(NCommon::TRequest& req) {
        const TStringBuf method = req.GetRequestMethod();

        auto it = HandlersMethod_.find(method);
        if (it == HandlersMethod_.end()) {
            throw TBadRequest() << "Unexpected method: " << req.GetRequestMethod();
        }

        TStringBuf path = req.GetPath();
        path.ChopSuffix("/");

        UnistatHandles_.Inc(BuildHandleKey(method, path));

        it->second(req, path);
    }

    void TSezamApi::ProcessGet(NCommon::TRequest& req, TStringBuf path) {
        auto it = HandlersGet_.find(path);
        if (it == HandlersGet_.end()) {
            throw TBadRequest() << "Unexpected path (GET): " << req.GetPath();
        }

        it->second(req);
    }

    void TSezamApi::ProcessPost(NCommon::TRequest& req, TStringBuf path) {
        auto it = HandlersPost_.find(path);
        if (it == HandlersPost_.end()) {
            throw TBadRequest() << "Unexpected path (POST): " << req.GetPath();
        }

        it->second(req);
    }

    void TSezamApi::ProcessOptions(NCommon::TRequest& req, TStringBuf path) {
        if (path == "/accounts" || path == "/all_accounts") {
            DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsForAccounts.get());
            return;
        }
        if (path == "/registration_status/check") {
            DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsForRegCompletionRecommended.get());
            return;
        }

        DefaultProcessing(req);
    }

    static const TString DBPOOL_STATS = "dbpoolstats";
    static const TString TITLE = "Lah monitoring status";
    void TSezamApi::ProcessScriptNagios(NCommon::TRequest& req) {
        if (req.HasArg(DBPOOL_STATS)) {
            NDbPool::TDbPoolStats stats(TITLE);
            stats.Add(&Runtime_->Bb());
            req.SetStatus(HTTP_OK);
            req.Write(TString(stats.Result()));
            return;
        }

        if (Runtime_->IsForceDown()) {
            TLog::Warning() << "Request failed: externally forced out-of-service";
            req.SetStatus(HTTP_SERVICE_UNAVAILABLE);
            req.Write("Externally forced out-of-service");
            return;
        }

        TString err;
        if (!Runtime_->IsOk(err)) {
            TLog::Error() << "lah nagios error: " << err;
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(TString("lah nagios error: " + err));
            return;
        }

        req.SetStatus(HTTP_OK);
        req.Write("Lah status is OK");
    }

    static const TString COOKIE_LAH = "lah";
    static const TString COOKIE_IGNORE_LAH_UNTIL = "ilahu";
    static const TString CGI_MULTI = "multi";
    static const TString CGI_POPUP = "popup";
    void TSezamApi::ProcessScriptSuggest(NCommon::TRequest& req) {
        DefaultProcessing(req);

        if (req.GetArg(CGI_POPUP) == "yes") {
            const time_t now = time(nullptr);
            const time_t ilahu = TUtils::GetTsFromCookie(req.GetCookie(COOKIE_IGNORE_LAH_UNTIL));
            if (now < ilahu) {
                req.Write(TSuggestProcessor::Fallback());
                TUtils::LogIgnore(req, TStringBuilder() << "suggest: now(" << now << ") < ilahu(" << ilahu << ")");
                return;
            }
        }

        req.Write(CreateSuggestProcessor(req).Process(
            req.GetCookie(COOKIE_LAH),
            req.GetArg(CGI_MULTI) == "yes"
                ? TSuggestProcessor::ENeed::All
                : TSuggestProcessor::ENeed::One));
    }

    static const TString CGI_POSTPONE_DELAY = "postpone_delay";
    static const TString RET_OK = R"({"status":"OK"}
)";
    void TSezamApi::ProcessScriptCloseSuggest(NCommon::TRequest& req) {
        DefaultProcessing(req);

        if (!CheckCsrf(req)) {
            req.Write(RET_OK);
            return;
        }

        ui32 delay = TUtils::GetPostponeDelay(
            req.GetArg(CGI_POSTPONE_DELAY),
            Runtime_->GetTimingsCfg().DoNotDisturbAfterCloseMin,
            Runtime_->GetTimingsCfg().DoNotDisturbAfterCloseMax,
            Runtime_->GetTimingsCfg().DoNotDisturbAfterClose);

        SetHeaderIlahu(req,
                       TUtils::GetTsFromCookie(req.GetCookie(COOKIE_IGNORE_LAH_UNTIL)),
                       delay);
        req.Write(RET_OK);
    }

    void TSezamApi::ProcessScriptSuggestShowAndPostpone(NCommon::TRequest& req) {
        DefaultProcessing(req);

        if (!CheckCsrf(req)) {
            req.Write(TSuggestProcessor::Fallback());
            return;
        }

        const time_t now = time(nullptr);
        const time_t ilahu = TUtils::GetTsFromCookie(req.GetCookie(COOKIE_IGNORE_LAH_UNTIL));
        if (now < ilahu) {
            req.Write(TSuggestProcessor::Fallback());
            TUtils::LogIgnore(req, TStringBuilder() << "suggest_postpone: now(" << now << ") < ilahu(" << ilahu << ")");
            return;
        }

        TSuggestProcessor p = CreateSuggestProcessor(req);
        const TString& res = p.Process(req.GetCookie(COOKIE_LAH),
                                       req.GetArg(CGI_MULTI) == "yes"
                                           ? TSuggestProcessor::ENeed::All
                                           : TSuggestProcessor::ENeed::One);

        if (p.IsSuccessful()) {
            ui32 delay = TUtils::GetPostponeDelay(
                req.GetArg(CGI_POSTPONE_DELAY),
                Runtime_->GetTimingsCfg().DoNotDisturbAfterShowMin,
                Runtime_->GetTimingsCfg().DoNotDisturbAfterShowMax,
                Runtime_->GetTimingsCfg().DoNotDisturbAfterShow);

            SetHeaderIlahu(req, ilahu, delay);
        }

        req.Write(res);
    }

    static const TString SESSION_ID_ = "Session_id";
    static const TString SESSGUARD_ = "sessguard";
    static const TString COOKIE_YANDEXUID = "yandexuid";

    void TSezamApi::ProcessScriptAccounts(NCommon::TRequest& req, EOnlyValidUsers onlyValidUsers) {
        DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsForAccounts.get());

        std::unique_ptr<NSezamApi::TAccounts> accounts;
        const TString& sess = req.GetCookie(SESSION_ID_);
        if (sess) {
            accounts = GetAccountsFromBb(
                Runtime_->Bb(),
                *AccountsSettings_,
                TAccountsArgs{
                    .Sessionid = sess,
                    .Sessguard = req.GetCookie(SESSGUARD_),
                    .Domain = req.GetHost(),
                    .Userip = req.GetRemoteAddr(),
                    .OnlyValidUsers = onlyValidUsers,
                    .AddNames = EAddNames::True,
                    .AddOrganizationInfo = EAddOrganizationInfo::True,
                });
        }

        req.Write(SerializeAccountsList(accounts.get()));
    }

    static const TString CGI_YU = "yu";
    bool TSezamApi::CheckCsrf(const NCommon::TRequest& req) {
        const TString& yuArg = req.GetArg(CGI_YU);
        const TString& yuCookie = req.GetCookie(COOKIE_YANDEXUID);

        if (yuCookie.empty() || yuCookie != yuArg) {
            TLog::Debug() << "Lah module CSRF check failed: yandexuid mismatch. Cgi <" << yuArg
                          << ">. Cookie <" << yuCookie << ">";
            return false;
        }
        return true;
    }

    static const TString HEADER_ORIGIN = "Origin";
    static const TString HEADER_ALLOW_ORIGIN = "Access-Control-Allow-Origin";
    static const TString HEADER_ALLOW_METHODS = "Access-Control-Allow-Methods";
    static const TString HEADER_ALLOW_METHODS_VAL = "GET, POST";
    static const TString HEADER_ALLOW_CREDENTIALS = "Access-Control-Allow-Credentials";
    static const TString HEADER_ALLOW_CREDENTIALS_VAL = "true";
    static const TString HEADER_ALLOW_HEADERS = "Access-Control-Allow-Headers";
    static const TString HEADER_ALLOW_HEADERS_VAL = "X-Requested-With";
    void TSezamApi::DefaultProcessing(NCommon::TRequest& req, const TOrigin* originChecker) {
        req.SetStatus(HTTP_OK);
        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);

        const TString& originHeader = req.GetHeader(HEADER_ORIGIN);
        std::optional<TString> parsedOrigin = Runtime_->GetOrigins().YandexTld->Check(originHeader);
        if (!parsedOrigin && originChecker) {
            parsedOrigin = originChecker->Check(originHeader);
        }
        if (!parsedOrigin) {
            return;
        }
        UnistatOrigins_.Add(*parsedOrigin);

        req.SetHeader(HEADER_ALLOW_ORIGIN, originHeader);
        req.SetHeader(HEADER_ALLOW_METHODS, HEADER_ALLOW_METHODS_VAL);
        req.SetHeader(HEADER_ALLOW_CREDENTIALS, HEADER_ALLOW_CREDENTIALS_VAL);
        req.SetHeader(HEADER_ALLOW_HEADERS, HEADER_ALLOW_HEADERS_VAL);
    }

    void TSezamApi::DefaultProcessing(NCommon::TRequest& req) {
        DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsDefault.get());
    }

    void TSezamApi::SetHeaderIlahu(NCommon::TRequest& req, time_t curIlahu, ui32 doNotDistrub) {
        const time_t now = time(nullptr);
        const time_t timeToSet = now + doNotDistrub;
        const time_t newValue = std::max(curIlahu, timeToSet);

        TLog::Debug() << "Set new value of ilahu=" << newValue
                      << ". yu=" << req.GetCookie(COOKIE_YANDEXUID);

        NCommon::TCookie ilahu(COOKIE_IGNORE_LAH_UNTIL, IntToString<10>(newValue));
        ilahu.SetHttpOnly(true);
        ilahu.SetExpires(now + Runtime_->GetTimingsCfg().IlahuTtl);
        ilahu.SetSecure(true);
        ilahu.SetDomain(TUtils::GetDomain(req.GetHost()));
        ilahu.SetSameSite(NCommon::TCookie::ESameSite::None);

        req.SetCookie(ilahu);
    }

    TSuggestProcessor TSezamApi::CreateSuggestProcessor(const NCommon::TRequest& request) const {
        return TSuggestProcessor(Runtime_->Bb(),
                                 request,
                                 Runtime_->GetMethodsCfg());
    }

    void TSezamApi::CheckGrants() {
        TSuggestProcessor::CheckBlackboxGrants(Runtime_->Bb());
        CheckAccountBlackboxGrants(Runtime_->Bb());
        CheckRegCompetionRecommendedBlackboxGrants(Runtime_->Bb());
    }

    void TSezamApi::ProcessScriptRegistrationStatusCheck(NCommon::TRequest& req) {
        DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsForRegCompletionRecommended.get());

        bool regCompletionRecommended = false;
        const TString& sess = req.GetCookie(SESSION_ID_);
        if (sess) {
            const std::optional<TString> ypCookie = req.HasCookie(COOKIE_YP_) ? std::optional<TString>(req.GetCookie(COOKIE_YP_)) : std::nullopt;
            regCompletionRecommended = TRegCompletionRecommended::CheckRegCompletionRecommended(
                Runtime_->Bb(),
                TRegCompletionArgs{
                    .Sessionid = sess,
                    .Sessguard = req.GetCookie(SESSGUARD_),
                    .Domain = req.GetHost(),
                    .Userip = req.GetRemoteAddr(),
                    .YPCookie = ypCookie,
                });
        }

        req.Write(TRegCompletionRecommended::SerializeRegCompletionRecommended(regCompletionRecommended));
    }

    void TSezamApi::ProcessScriptRegistrationStatusPostpone(NCommon::TRequest& req) {
        DefaultProcessing(req, Runtime_->GetOrigins().SideDomainsForRegCompletionRecommended.get());

        const TString& connectionIdFromReq = req.GetArg("ci"); // it should be in body, but it was filled in ScanCgiFromBody
        if (connectionIdFromReq.empty()) {
            TUtils::LogIgnore(req, "No ci in request");
            req.Write(TRegCompletionRecommended::DefaultAnswer());
            return;
        }

        const TString sess = req.GetCookie(SESSION_ID_);
        if (sess.empty()) {
            req.Write(TRegCompletionRecommended::DefaultAnswer());
            return;
        }

        const TString& connectionId = TRegCompletionRecommended::GetConnectionId(
            Runtime_->Bb(),
            TRegCompletionArgs{
                .Sessionid = sess,
                .Sessguard = req.GetCookie(SESSGUARD_),
                .Domain = req.GetHost(),
                .Userip = req.GetRemoteAddr(),
            });

        if (!connectionId.empty() && connectionId == connectionIdFromReq) {
            TRegCompletionRecommended::SetLrcsToYpCookie(req, Runtime_->GetTimingsCfg().LrcsExpirationTtl, TInstant::Now());
        } else {
            TUtils::LogIgnore(req, TStringBuilder() << "ci differs: " << connectionId << "(from bb) != " << connectionIdFromReq << "(from request)");
        }

        req.Write(TRegCompletionRecommended::DefaultAnswer());
    }
}
