#include "sids_accounts.h"

#include <passport/infra/daemons/sezamapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/blackbox2/blackbox2.h>
#include <library/cpp/geobase/lookup.hpp>

namespace NPassport::NSezamApi {
    TAccountsSettings::TAccountsSettings(const NXml::TConfig& config, const TString& path) {
        const TString geobasePath = config.AsString(path + "/geobase", "");
        if (geobasePath.empty()) {
            throw yexception() << "Geobase not configured!";
        }

        Lookup_ = std::make_unique<NGeobase::TLookup>(geobasePath.c_str());

        PlusForPdd_ = config.AsBool(path + "/plus_for_pdd", true);

        const TString domain_list = config.AsString(path + "/plus_domains", "");
        PlusDomains_ = NUtils::ToVector(domain_list, ",;");

        TLog::Info() << "Accounts settings: plus_for_pdd: " << (PlusForPdd_ ? "yes" : "no")
                     << ", plus_domains: " << domain_list;
    }

    bool TAccountsSettings::InPlusDomain(const TString& host) const {
        TString domain(host);
        TString::size_type pos = domain.rfind('.');
        if (pos > 0) {
            pos = domain.rfind('.', pos - 1);
            if (pos != TString::npos) {
                domain.erase(0, pos + 1);
            }
        }

        return std::find(PlusDomains_.begin(), PlusDomains_.end(), domain) != PlusDomains_.end();
    }

    bool TAccountsSettings::IsRussia(const TString& userip) const {
        if (!Lookup_) {
            throw yexception() << "Geobase not initialized!";
        }

        return Lookup_->IsIpInRegion(userip, 225);
    }

    TAccountsSettings::~TAccountsSettings() {
    }

    void TPersonalInfo::Serialize(NJson::TObject& wr) const {
        wr.Add("uid", Uid);
        wr.Add("login", Login);
        {
            NJson::TObject disp(wr, "displayName");
            disp.Add("name", DisplayName);
            if (FirstName) {
                disp.Add("firstname", *FirstName);
            }
            if (LastName) {
                disp.Add("lastname", *LastName);
            }
            disp.Add("default_avatar", DefaultAvatar);
            if (Social) {
                NJson::TObject social(disp, "social");
                social.Add("profileId", ProfileId);
                social.Add("provider", Provider);
            }
        }

        wr.Add("defaultEmail", DefaultEmail);

        NJson::TObject attrs(wr, "attributes");
        for (const auto& [key, value] : Attributes) {
            if (const TString* val = std::get_if<TString>(&value); val) {
                attrs.Add(key, *val);
            } else if (const bool* val = std::get_if<bool>(&value); val) {
                attrs.Add(key, *val);
            } else if (const ui32* val = std::get_if<ui32>(&value); val) {
                attrs.Add(key, *val);
            }
        }
    }

    bool plusAvailable(bool isChild,
                       const TAccountsSettings& settings,
                       const TString& userip, const TString& domain, bool isPdd) {
        // we successfully checked cookie for this domain so it is not a fraud one
        if (isChild) {
            return false;
        }

        if (!settings.InPlusDomain(domain))
            return false;

        if (isPdd && !settings.PlusForPdd())
            return false;

        if (!settings.IsRussia(userip))
            return false;

        return true;
    }

    static bool AttrAsBool(const TString& attrValue) {
        return !attrValue.empty() && attrValue[0] != '0';
    }

    static const TString ATTR_HAS_PLUS("1015");
    static const TString ATTR_FIRSTNAME("27");
    static const TString ATTR_LASTNAME("28");
    static const TString ATTR_ORGANIZATION_IDS("1017");
    static const TString ATTR_ACCOUNT_PROTECTION_LEVEL("1013");
    static const TString ATTR_CONFIRMED_PHONE("110");
    static const TString ATTR_HAS_2FA("1003");
    static const TString ATTR_IS_CHILD("210");
    static const TString ATTR_CONTENT_RATING("202");

    static const TString BB_QUERY_PATH = "/blackbox";
    static const TString METHOD_POST = "POST";
    static NDbPool::TQuery BuildBbQuery(const TAccountsArgs& args) {
        NBlackbox2::TOptions opt(NBlackbox2::OPT_REGNAME);
        opt << NBlackbox2::OPT_MULTISESSION << NBlackbox2::OPT_FULL_INFO;
        opt << NBlackbox2::TOption("dbfields", "subscription.login.668,subscription.login.669") << NBlackbox2::OPT_GET_DEFAULT_EMAIL;

        NBlackbox2::TAttributes attrs(ATTR_HAS_PLUS);
        if (args.AddNames == EAddNames::True) {
            attrs << ATTR_FIRSTNAME;
            attrs << ATTR_LASTNAME;
        }
        if (args.AddOrganizationInfo == EAddOrganizationInfo::True) {
            attrs << ATTR_ORGANIZATION_IDS;
        }

        attrs << ATTR_ACCOUNT_PROTECTION_LEVEL;
        attrs << ATTR_CONFIRMED_PHONE;
        attrs << ATTR_HAS_2FA;
        attrs << ATTR_IS_CHILD;
        attrs << ATTR_CONTENT_RATING;

        opt << attrs;

        NDbPool::TQuery query(BB_QUERY_PATH);
        query.SetHttpMethod(METHOD_POST);
        query.SetHttpBody(
            NUtils::CreateStr(
                NBlackbox2::SessionIDRequest(args.Sessionid, args.Domain, args.Userip, opt),
                "&sessguard=",
                NUtils::Urlencode(args.Sessguard),
                "&allow_child=yes"));
        return query;
    }

    static const TStringBuf PROTECTION_LEVEL_RED = "4";
    static const TStringBuf PROTECTION_LEVEL_YELLOW = "16";

    std::unique_ptr<TAccounts> GetAccountsFromBb(NDbPool::TDbPool& db,
                                                 const TAccountsSettings& settings,
                                                 const TAccountsArgs& args) {
        try {
            const TString body = NDbPool::NUtils::GetHttpBody(db, BuildBbQuery(args), "Blackbox error");
            THolder<NBlackbox2::TMultiSessionResp> resp = NBlackbox2::SessionIDResponseMulti(body);

            if (resp->Status() != NBlackbox2::TSessionResp::Valid && resp->Status() != NBlackbox2::TSessionResp::NeedReset) {
                return {};
            }

            NBlackbox2::TAllowMoreUsers can_more(resp.Get());

            std::unique_ptr<TAccounts> ret = std::make_unique<TAccounts>(resp->Count(), can_more.AllowMoreUsers());
            ret->DefaultUid = resp->DefaultUid();

            for (int i = 0; i < resp->Count(); ++i) {
                THolder<NBlackbox2::TSessionResp> user = resp->User(i);
                TPersonalInfo& info = ret->Users[i];

                const bool isUserValid = user->Status() == NBlackbox2::TSessionResp::Valid ||
                                         user->Status() == NBlackbox2::TSessionResp::NeedReset;

                NBlackbox2::TUid uid(user.Get());
                if (args.OnlyValidUsers == EOnlyValidUsers::True && !isUserValid) {
                    TLog::Debug() << "SidsModule/accounts: uid is not valid: " << uid.Uid()
                                  << ". status is " << static_cast<int>(user->Status());
                    continue;
                }

                NBlackbox2::TLoginInfo login(user.Get());
                NBlackbox2::TDisplayNameInfo dname(user.Get());
                NBlackbox2::TDBFields fields(user.Get());
                NBlackbox2::TEmailList emails(user.Get());
                NBlackbox2::TAttributes attrs(user.Get());

                info.Uid = resp->Id(i);
                info.Login = login.Login();

                info.DisplayName = dname.Name();
                info.Social = dname.Social();
                info.ProfileId = dname.SocProfile();
                info.Provider = dname.SocProvider();
                info.DefaultAvatar = dname.DefaultAvatar();

                info.DefaultEmail = emails.GetDefault();

                for (const auto& pair : fields) {
                    const TString& fieldName = pair.first;
                    const TString& fieldValue = pair.second;

                    if (!fieldName.StartsWith("subscription.login.") || fieldValue.empty())
                        continue; // skip alien dbfields or empty values (i.e. no subscription)

                    TString sid = fieldName.substr(fieldName.rfind('.') + 1);

                    if (sid == "668") {
                        info.Attributes["beta-tester"] = true;
                    } else if (sid == "669") {
                        info.Attributes["staff"] = true;
                        info.Attributes["staff-login"] = fieldValue;
                    }
                }

                const bool isChild = AttrAsBool(attrs.Get(ATTR_IS_CHILD));
                if (isChild) {
                    info.Attributes["is_child"] = true;
                }

                const TString& hasPlus = attrs.Get(ATTR_HAS_PLUS);
                if (AttrAsBool(hasPlus)) {
                    info.Attributes["has_plus"] = true;
                } else if (!uid.Uid().empty() && plusAvailable(isChild, settings, args.Userip, args.Domain, uid.Hosted())) {
                    info.Attributes["plus_available"] = true;
                }

                if (args.AddNames == EAddNames::True && isUserValid) {
                    info.FirstName = attrs.Get(ATTR_FIRSTNAME);
                    info.LastName = attrs.Get(ATTR_LASTNAME);
                }

                if (isUserValid) {
                    if (attrs.Get(ATTR_HAS_2FA).empty()) {
                        info.Attributes["missing_2fa"] = true;
                    }
                    if (attrs.Get(ATTR_CONFIRMED_PHONE).empty()) {
                        info.Attributes["missing_confirmed_phone"] = true;
                    }

                    const TString& protectionLevel = attrs.Get(ATTR_ACCOUNT_PROTECTION_LEVEL);
                    if (protectionLevel == PROTECTION_LEVEL_RED || protectionLevel == PROTECTION_LEVEL_YELLOW) {
                        info.Attributes["weak_account_protection"] = true;
                    }
                }

                if (args.AddOrganizationInfo == EAddOrganizationInfo::True && !attrs.Get(ATTR_ORGANIZATION_IDS).empty()) {
                    info.Attributes["has_organization"] = true;
                }

                if (const TString& rating = attrs.Get(ATTR_CONTENT_RATING); rating) {
                    ui32 num = 0;
                    if (TryIntFromString<10>(rating, num)) {
                        info.Attributes["content_rating"] = num;
                    } else {
                        TLog::Error() << "SidsModule/accounts: content_rating is not a number: '" << rating
                                      << "' on uid=" << info.Uid;
                    }
                }
            }

            for (const TPersonalInfo& p : ret->Users) {
                if (!p.Uid.empty()) {
                    return ret;
                }
            }

            return {};

        } catch (const std::exception& e) {
            TLog::Warning() << "Error: SidsModule/accounts: failed to get data from blackbox: " << e.what();
            throw TBackendException() << e.what();
        }
    }

    void CheckAccountBlackboxGrants(NDbPool::TDbPool& bb) {
        NDbPool::TQuery query = BuildBbQuery(TAccountsArgs{
            .AddNames = EAddNames::True,
            .AddOrganizationInfo = EAddOrganizationInfo::True,
        });

        TUtils::CheckBlackboxGrants(bb, std::move(query));
    }

    TString SerializeAccountsList(const TAccounts* accounts) {
        TString body;
        NJson::TWriter wr(body);
        NJson::TObject root(wr);

        if (accounts) {
            root.Add("default_uid", accounts->DefaultUid);
            root.Add("can-add-more", accounts->CanMore);

            NJson::TArray arr(root, "accounts");
            for (const TPersonalInfo& p : accounts->Users) {
                if (p.Uid.empty())
                    continue;
                NJson::TObject obj(arr);
                p.Serialize(obj);
            }
        }

        return body;
    }
}
