#include "suggest_processor.h"

#include "cookie_parser.h"

#include <passport/infra/daemons/sezamapi/src/utils/fetcher.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NSezamApi {
    static const TString COOKIE_YANDEXUID = "yandexuid";
    static const TString FALLBACK = R"({"accounts":[]})";

    TSuggestProcessor::TSuggestProcessor(NDbPool::TDbPool& bb,
                                         const NCommon::TRequest& request,
                                         const TRuntimeContext::TMethodsConfig& methodsMapping)
        : Bb_(bb)
        , Req_(request)
        , MethodsMapping_(methodsMapping)
    {
    }

    TString TSuggestProcessor::Process(TStringBuf cookie, ENeed need) {
        if (cookie.empty()) {
            TUtils::LogIgnore(Req_, "cookie 'lah' is empty");
            return FALLBACK;
        }

        TLahCookieParser::TParseResult result = TLahCookieParser::ParseV2(
            TBbCheckSignFetcher(Bb_), cookie);

        if (result.ErrorMessage) {
            TLog::Debug() << "Failed to parse cookie: " << result.ErrorMessage
                          << ". Body: " << cookie
                          << ". yu=" << Req_.GetCookie(COOKIE_YANDEXUID);
            return FALLBACK;
        }

        if (result.Accounts.empty()) {
            TLog::Error() << "Cookie contains empty proto. Body: " << cookie
                          << ". yu=" << Req_.GetCookie(COOKIE_YANDEXUID);
            return FALLBACK;
        }

        try {
            TAccountsInfo info = TBbAccountsFetcher(Bb_).Fetch(result.Accounts);

            Res_ = SerializeJson(result.Accounts, info, need, MethodsMapping_, Req_.GetCookie(COOKIE_YANDEXUID));
            if (Res_.empty()) {
                TUtils::LogIgnore(Req_, "impossible case");
                return FALLBACK;
            }

            return Res_;
        } catch (const std::exception& e) {
            TLog::Debug() << "Failed to parse request from blackbox: " << e.what()
                          << ". yu=" << Req_.GetCookie(COOKIE_YANDEXUID);
            return FALLBACK;
        }
    }

    bool TSuggestProcessor::IsSuccessful() const {
        return !Res_.empty() && Res_ != FALLBACK;
    }

    void TSuggestProcessor::CheckBlackboxGrants(NDbPool::TDbPool& bb) {
        TBbAccountsFetcher(bb).CheckGrants();
        TBbCheckSignFetcher(bb).CheckGrants();
    }

    const TString& TSuggestProcessor::Fallback() {
        return FALLBACK;
    }

    TString TSuggestProcessor::SerializeJson(const TLahAccounts& accounts,
                                             const TAccountsInfo& info,
                                             ENeed need,
                                             const TRuntimeContext::TMethodsConfig& methodsMapping,
                                             const TString& yu) {
        TStringBuilder notFoundUids;
        TStringBuilder glogoutUids;
        TStringBuilder validUids;

        TString res;
        NJson::TWriter wr(res);
        NJson::TObject root(wr);
        NJson::TArray arr(root, "accounts");

        for (const TLahAccount& lahAcc : accounts) {
            auto itAcc = info.find(IntToString<10>(lahAcc.Uid));
            if (itAcc == info.end()) {
                notFoundUids << lahAcc.Uid << ',';
                continue;
            }

            if (itAcc->Glogout > lahAcc.LastLoginTime || itAcc->RevokeSessions > lahAcc.LastLoginTime) {
                glogoutUids << lahAcc.Uid << ',';
                continue;
            }

            validUids << lahAcc.Uid << ',';

            NJson::TObject obj(arr);
            obj.Add("display_name", itAcc->DisplayName);
            obj.Add("uid", lahAcc.Uid);

            auto it = methodsMapping.Mapping.find(lahAcc.LoginMethod);
            if (it == methodsMapping.Mapping.end()) {
                obj.Add("method", nullptr);
            } else {
                obj.Add("method", it->second);
            }

            NJson::TObject ava(obj, "avatar");
            ava.Add("default", itAcc->Avatar);
            ava.Add("empty", itAcc->IsEmpty);

            if (need == ENeed::One) {
                break;
            }
        }

        TLog::Debug() << "Request proccessed: valid=" << validUids
                      << " glogout=" << glogoutUids
                      << " not_found=" << notFoundUids
                      << " yu=" << yu;

        return res;
    }
}
