#include "fetcher.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

namespace NPassport::NSezamApi {
    TString TBbCheckSignFetcher::ParseBbResponse(TStringBuf data) const {
        rapidjson::Document document;
        Y_ENSURE(NJson::TReader::DocumentAsObject(data, document), "Failed to parse JSON");

        TString status;
        Y_ENSURE(NJson::TReader::MemberAsString(document, "status", status), "Missing status field");
        Y_ENSURE(status == "OK", "Bad check_sign status: " << status);

        TString value;
        Y_ENSURE(NJson::TReader::MemberAsString(document, "value", value) && value, "Missing value field");

        return value;
    }

    static const TString BB_CHECK_SIGN_QUERY_PATH = "/blackbox";
    static const TString BB_CHECK_SIGN_METHOD = "POST";
    NDbPool::TQuery TBbCheckSignFetcher::BuildBbQuery(const TStringBuf& data) const {
        NDbPool::TQuery query(BB_CHECK_SIGN_QUERY_PATH);
        query.SetHttpMethod(BB_CHECK_SIGN_METHOD);
        query.SetHttpBody(
            NUtils::CreateStr(
                "method=check_sign&format=json&sign_space=lah&signed_value=",
                NUtils::Urlencode(data)));
        return query;
    }

    static const rapidjson::Pointer UID_POINTER("/uid/value");
    TAccountsInfo TBbAccountsFetcher::ParseBbResponse(TStringBuf data) const {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(data, doc)) {
            TLog::Error() << "Invalid json from blackbox: " << data;
            return {};
        }

        const rapidjson::Value* jsonUsers = nullptr;
        if (!NJson::TReader::MemberAsArray(doc, "users", jsonUsers)) {
            TLog::Error() << "Unexpected format from blackbox ('users'): " << data;
            return {};
        }

        TAccountsInfo res;

        for (size_t idx = 0; idx < jsonUsers->Size(); ++idx) {
            const rapidjson::Value& jsonUser = (*jsonUsers)[idx];
            const rapidjson::Value* jsonUid = UID_POINTER.Get(jsonUser);
            if (!jsonUid) {
                continue;
            }

            TString dispName;
            TString avatar;
            bool isAvaEmpty = false;
            {
                const rapidjson::Value* jsonDispNameObj = nullptr;
                const rapidjson::Value* jsonAvatarObj = nullptr;
                if (!NJson::TReader::MemberAsObject(jsonUser, "display_name", jsonDispNameObj) ||
                    !NJson::TReader::MemberAsObject(*jsonDispNameObj, "avatar", jsonAvatarObj) ||
                    !NJson::TReader::MemberAsString(*jsonDispNameObj, "name", dispName) ||
                    !NJson::TReader::MemberAsString(*jsonAvatarObj, "default", avatar) ||
                    !NJson::TReader::MemberAsBool(*jsonAvatarObj, "empty", isAvaEmpty)) {
                    TLog::Error() << "Unexpected format from blackbox ('display_name'): " << data;
                    return {};
                }
            }

            time_t glogout = 0;
            time_t revoke = 0;
            {
                const rapidjson::Value* jsonAttrsObj = nullptr;
                if (!NJson::TReader::MemberAsObject(jsonUser, "attributes", jsonAttrsObj)) {
                    TLog::Error() << "Unexpected format from blackbox ('attributes'): " << data;
                    return {};
                }

                TString s;
                if (NJson::TReader::MemberAsString(*jsonAttrsObj, "4", s)) {
                    glogout = IntFromString<time_t, 10>(s);
                }
                if (NJson::TReader::MemberAsString(*jsonAttrsObj, "136", s)) {
                    revoke = IntFromString<time_t, 10>(s);
                }
            }

            res.insert(TAccountInfo{
                .Uid = TString(jsonUid->GetString(), jsonUid->GetStringLength()),
                .DisplayName = std::move(dispName),
                .Avatar = std::move(avatar),
                .IsEmpty = isAvaEmpty,
                .Glogout = glogout,
                .RevokeSessions = revoke,
            });
        }

        return res;
    }

    static const TString BB_ACCOUNTS_QUERY_PATH = "/blackbox";
    static const TString BB_ACCOUNTS_METHOD = "POST";
    static const TString BB_ACCOUNTS_CGI = "attributes=4,136&format=json&method=userinfo&regname&userip=0.0.0.0&uid=";
    NDbPool::TQuery TBbAccountsFetcher::BuildBbQuery(const TLahAccounts& accounts) const {
        TStringStream request;
        request.Reserve(BB_ACCOUNTS_CGI.Size() + 15 * accounts.size());

        request << BB_ACCOUNTS_CGI;
        for (const TLahAccount& acc : accounts) {
            request << acc.Uid << ',';
        }

        if (request.Str().EndsWith(',')) {
            request.Str().pop_back();
        }

        NDbPool::TQuery q(BB_ACCOUNTS_QUERY_PATH);
        q.SetHttpMethod(BB_ACCOUNTS_METHOD);
        q.SetHttpBody(std::move(request.Str()));
        return q;
    }
}
