#include "origin.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/yexception.h>

namespace NPassport::NSezamApi {
    TOriginPtr TOrigin::CreateForYandexTld(EPolicy policy, const TString& sld, const TDomains& domains) {
        Y_ENSURE(!domains.empty(), "List of yandex_tld is empty");
        return std::make_unique<TOrigin>(RegexForYandexTld(policy, sld, domains), sld + ".$1");
    }

    TOriginPtr TOrigin::CreateForSideDomains(TOrigin::EPolicy policy, const TDomains& domains) {
        if (domains.empty()) {
            return {};
        }

        return std::make_unique<TOrigin>(RegexForSideDomains(policy, domains), "$1");
    }

    std::optional<TString> TOrigin::Check(const TStringBuf origin) const {
        TString res;
        if (!Regex_.Apply(origin, res)) {
            return {};
        }

        return res;
    }

    TString TOrigin::RegexForYandexTld(EPolicy policy, const TString& sld, const TDomains& domains) {
        Y_ENSURE(!domains.empty());

        TString res = NUtils::CreateStr(
            "^https",
            policy == EPolicy::Strict ? "" : "?",
            R"(://(?:[\w_\-.]+\.)?)",
            sld,
            R"(\.()");

        for (const TString& dom : domains) {
            res.append(NUtils::ReplaceAny(dom, ".", R"(\.)")).push_back('|');
        }

        res.pop_back();
        res.append(policy == EPolicy::Strict ? ")$" : R"()(:[\d]{1,5})?$)");

        return res;
    }

    TString TOrigin::RegexForSideDomains(TOrigin::EPolicy policy, const TDomains& domains) {
        Y_ENSURE(!domains.empty());
        TString res = NUtils::CreateStr(
            "^https",
            policy == EPolicy::Strict ? "" : "?",
            R"(://(?:[\w_\-.]+\.)?()");

        for (const TString& dom : domains) {
            res.append(NUtils::ReplaceAny(dom, ".", R"(\.)")).push_back('|');
        }

        res.pop_back();
        res.append(policy == EPolicy::Strict ? ")$" : R"()(:[\d]{1,5})?$)");

        return res;
    }

    TOrigin::TOrigin(TStringBuf regex, TStringBuf outFormat)
        : Regex_(regex, outFormat)
    {
    }
}
