#include <passport/infra/daemons/sezamapi/src/cookie_parser.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/output.h>

#include <unordered_map>

using namespace NPassport;
using namespace NPassport::NSezamApi;
using namespace NPassport::NDbPool;

template <>
void Out<TLahAccount>(IOutputStream& o, const TLahAccount& value) {
    o << "(uid: " << value.Uid << ", glogout: " << value.LastLoginTime << ", method: " << value.LoginMethod << ")";
}

template <>
void Out<TLahAccounts>(IOutputStream& o, const TLahAccounts& value) {
    for (const auto& a : value) {
        o << a << Endl;
    }
}

Y_UNIT_TEST_SUITE(ParserV2) {
    static const std::unordered_map<TString, TString> CHECK_SIGN_VALUES{
        {"asdf", "fdas"},
        {"afasdf", "fd%s"},
        {"1623400017.310308.sVzUQjmRigngvkW5.bo0YSGoLMVdPBDQ8iw9NofiJ7CzuFOvlg10HvJO6hO4Nbtuhz_pK9w.6UmaJG073IdOsRv0t3RCqQ", "Cg0IARC5tZcLGJSRBiABCg0IARDoo9UTGJWRBiAC"},
        {"1623399943.310308.aJ5bvtA0duqu634d.ksTc73EbWhYMeIi6kEw.Yiz5K9DMvvUXdGonBLMWaQ", "CggIARABGAIgAw"},
        {"1623399881.310308.0fygQnIwo0WJBtn-.h0DHN7-zs1X2L-r5p_xT6T21XiM3ML5k3rt4cwPeiAV7gZGUa6WwNPfFhmRxT3RYkaoH2XyDhSrWsUsyaNOzQDuaPsiqBDv2xOnJ.f6Kg44BwGHw4-aldpTn3ug", "CgwIARDkpgQYlJEGIAEKDAgBEOWmBBiVkQYgAQoMCAEQ5qYEGJaRBiAACgwIARDnpgQYl5EGIAA"},
    };

    class TFakeCheckSignFetcher: public TBaseCheckSignFetcher {
    public:
        TString Fetch(const TStringBuf& data) const override {
            auto it = CHECK_SIGN_VALUES.find(TString(data));
            if (it == CHECK_SIGN_VALUES.end()) {
                throw std::runtime_error("Failed to check signature");
            }
            return it->second;
        }
    };

    class TBbCheckSignFetcherTest: TBbCheckSignFetcher {
    public:
        TBbCheckSignFetcherTest(NDbPool::TDbPool& bb)
            : TBbCheckSignFetcher(bb)
        {
        }
        using TBbCheckSignFetcher::BuildBbQuery;
        using TBbCheckSignFetcher::ParseBbResponse;
    };

    class TBbCheckSignFetcherHolder {
    public:
        TBbCheckSignFetcherHolder()
            : Bb((NDbPool::TDbPoolSettings{
                  .Dsn = NDbPool::TDestination::CreateHttp(),
              }))
            , Fetcher(Bb)
        {
        }

        NDbPool::TDbPool Bb;
        TBbCheckSignFetcherTest Fetcher;
    };

    Y_UNIT_TEST(ParseBadData) {
        struct TTestCase {
            TString In;
            TString Exception;
        };

        std::vector<TTestCase> tests{
            TTestCase{
                .In = "asdf:fda",
                .Exception = "Malformed cookie"},
            TTestCase{
                .In = "2:asdf:fda",
                .Exception = "Failed to check signature"},
            TTestCase{
                .In = "2:fda",
                .Exception = "Failed to check signature"},
            TTestCase{
                .In = "2:afasdf",
                .Exception = "Bad base64url in cookie"},
            TTestCase{
                .In = "2:asdf",
                .Exception = "Malformed protobuf in cookie"},
        };

        for (const TTestCase& test : tests) {
            auto exc = TLahCookieParser::ParseV2(TFakeCheckSignFetcher(), test.In).ErrorMessage;
            UNIT_ASSERT_VALUES_EQUAL(TLahCookieParser::ParseV2(TFakeCheckSignFetcher(), test.In).ErrorMessage, test.Exception);
        }
    }

    Y_UNIT_TEST(ParseGoodData) {
        struct TTestCase {
            TLahAccounts Accounts;
            TString Cookie;
        };

        std::vector<TTestCase> tests{
            TTestCase{
                .Accounts = {
                    TLahAccount{
                        .Uid = 1,
                        .LastLoginTime = 2,
                        .LoginMethod = 3,
                    },
                },
                .Cookie = "2:1623399943.310308.aJ5bvtA0duqu634d.ksTc73EbWhYMeIi6kEw.Yiz5K9DMvvUXdGonBLMWaQ",
            },
            TTestCase{
                .Accounts = {
                    TLahAccount{
                        .Uid = 23452345,
                        .LastLoginTime = 100500,
                        .LoginMethod = 1,
                    },
                    TLahAccount{
                        .Uid = 41243112,
                        .LastLoginTime = 100501,
                        .LoginMethod = 2,
                    },
                },
                .Cookie = "2:1623400017.310308.sVzUQjmRigngvkW5.bo0YSGoLMVdPBDQ8iw9NofiJ7CzuFOvlg10HvJO6hO4Nbtuhz_pK9w.6UmaJG073IdOsRv0t3RCqQ",
            },
            TTestCase{
                .Accounts = {
                    TLahAccount{
                        .Uid = 70500,
                        .LastLoginTime = 100500,
                        .LoginMethod = 1,
                    },
                    TLahAccount{
                        .Uid = 70501,
                        .LastLoginTime = 100501,
                        .LoginMethod = 1,
                    },
                    TLahAccount{
                        .Uid = 70502,
                        .LastLoginTime = 100502,
                        .LoginMethod = 0,
                    },
                    TLahAccount{
                        .Uid = 70503,
                        .LastLoginTime = 100503,
                        .LoginMethod = 0,
                    },
                },
                .Cookie = "2:1623399881.310308.0fygQnIwo0WJBtn-.h0DHN7-zs1X2L-r5p_xT6T21XiM3ML5k3rt4cwPeiAV7gZGUa6WwNPfFhmRxT3RYkaoH2XyDhSrWsUsyaNOzQDuaPsiqBDv2xOnJ.f6Kg44BwGHw4-aldpTn3ug",
            },
        };

        for (const auto& test : tests) {
            TLahCookieParser::TParseResult result = TLahCookieParser::ParseV2(TFakeCheckSignFetcher(), test.Cookie);
            UNIT_ASSERT_STRINGS_EQUAL(result.ErrorMessage, "");
            UNIT_ASSERT_VALUES_EQUAL(result.Accounts.size(), test.Accounts.size());
            UNIT_ASSERT_VALUES_EQUAL(result.Accounts, test.Accounts);
        }
    }

    Y_UNIT_TEST(BbQuery) {
        TBbCheckSignFetcherHolder f;
        NDbPool::TQuery q = f.Fetcher.BuildBbQuery("a _^afdJNjskaj+=!@#$%^");
        UNIT_ASSERT_VALUES_EQUAL(q.HttpMethod(), "POST");
        UNIT_ASSERT_STRINGS_EQUAL(q.HttpBody(), "method=check_sign&format=json&sign_space=lah&signed_value=a+_^afdJNjskaj%2B%3D%21%40%23%24%25^");
    }

    Y_UNIT_TEST(ParseBbResponse) {
        TBbCheckSignFetcherHolder f;
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({)"), yexception, "Failed to parse JSON");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({})"), yexception, "Missing status field");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({"value": "asf"})"), yexception, "Missing status field");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({"status": "OK"})"), yexception, "Missing value field");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({"status": "OK", "value": ""})"), yexception, "Missing value field");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({"status": "NOT_OK", "value": "asf"})"), yexception, "Bad check_sign status");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f.Fetcher.ParseBbResponse(R"({"status": "NOT_OK"})"), yexception, "Bad check_sign status");

        UNIT_ASSERT_VALUES_EQUAL(f.Fetcher.ParseBbResponse(R"({"status": "OK", "value": "asdf"})"), "asdf");
    }

    Y_UNIT_TEST(acc) {
        TLahAccounts set = {
            {70500, 100500, true},
            {70501, 100501, true},
            {70502, 100502, false},
            {70503, 100503, false}};

        UNIT_ASSERT_EQUAL(4, set.size());
        auto it = set.begin();
        UNIT_ASSERT_EQUAL(TLahAccount({70501, 100501, true}), *it);
        ++it;
        UNIT_ASSERT_EQUAL(TLahAccount({70500, 100500, true}), *it);
        ++it;
        UNIT_ASSERT_EQUAL(TLahAccount({70503, 100503, false}), *it);
        ++it;
        UNIT_ASSERT_EQUAL(TLahAccount({70502, 100502, false}), *it);
    }
}
