#include <passport/infra/daemons/sezamapi/src/cookie_parser.h>
#include <passport/infra/daemons/sezamapi/src/suggest_processor.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NSezamApi;

Y_UNIT_TEST_SUITE(Suggest) {
    class TTestBbAccountFetcher: public TBbAccountsFetcher {
    public:
        TTestBbAccountFetcher(NDbPool::TDbPool& bb)
            : TBbAccountsFetcher(bb)
        {
        }

        using TBbAccountsFetcher::BuildBbQuery;
        using TBbAccountsFetcher::ParseBbResponse;
    };

    class TBbAccountFetcherHolder {
    public:
        TBbAccountFetcherHolder()
            : Bb(NDbPool::TDbPoolSettings{
                  .Dsn = NDbPool::TDestination::CreateHttp(),
              })
            , Fetcher(Bb)
        {
        }

        NDbPool::TDbPool Bb;
        TTestBbAccountFetcher Fetcher;
    };

    static const TString BB_RESP = R"(
    {
      "users": [
        {
         "karma_status": {
           "value": 0
         },
         "karma": {
           "value": 0
         },
         "uid": {},
         "id": "123000000000"
        },
        {
          "attributes": {
            "4": "100500",
            "136": "118"
          },
          "display_name": {
            "avatar": {
              "empty": false,
              "default": "1132/4564654-12313"
            },
            "name": "😂 Я 'Тестовый\" / &<тестер>"
          },
          "id": "70500",
          "uid": {
            "hosted": false,
            "lite": false,
            "value": "70500"
          },
          "login": "Junit-Test",
          "have_password": true,
          "have_hint": true,
          "karma": {
            "value": 0
          },
          "karma_status": {
            "value": 0
          },
          "regname": "Junit-Test"
        },
        {
          "attributes": {
            "4": "1",
            "136": "179"
          },
          "display_name": {
            "avatar": {
              "empty": true,
              "default": "0/0-0"
            },
            "name": "yandex-team-mm-15040"
          },
          "id": "7400",
          "uid": {
            "hosted": false,
            "lite": false,
            "value": "7400"
          },
          "login": "yandex-team-mm-15040",
          "have_password": true,
          "have_hint": true,
          "karma": {
            "value": 0
          },
          "karma_status": {
            "value": 0
          },
          "regname": "yandex-team-mm-15040"
        }
      ]
    })";

    Y_UNIT_TEST(parseBbResponse) {
        TBbAccountFetcherHolder h;
        TAccountsInfo info = h.Fetcher.ParseBbResponse(BB_RESP);
        UNIT_ASSERT_EQUAL(2, info.size());

        UNIT_ASSERT_EQUAL(TAccountInfo({"70500",
                                        R"(😂 Я 'Тестовый" / &<тестер>)",
                                        R"(1132/4564654-12313)",
                                        false,
                                        100500,
                                        118}),
                          *info.begin());

        UNIT_ASSERT_EQUAL(TAccountInfo({"7400",
                                        R"(yandex-team-mm-15040)",
                                        R"(0/0-0)",
                                        true,
                                        1,
                                        179}),
                          *info.rbegin());
    }

    Y_UNIT_TEST(parseBbResponseErr) {
        TBbAccountFetcherHolder h;
        UNIT_ASSERT(h.Fetcher.ParseBbResponse(
                                 R"({"users":[)")
                        .empty());
        UNIT_ASSERT(h.Fetcher.ParseBbResponse(
                                 R"({"users":["uid":{"value":"1"},"display_name":[]]})")
                        .empty());
        UNIT_ASSERT(h.Fetcher.ParseBbResponse(
                                 R"({"users":["uid":{"value":"1"},"display_name":{"avatar":{"empty":true,"default":"0/0-0"},"name": "yandex"}]})")
                        .empty());
    }

    Y_UNIT_TEST(buildBbRequest) {
        TBbAccountFetcherHolder h;
        TLahAccounts accs{
            TLahAccount{
                .Uid = 123,
            },
            TLahAccount{
                .Uid = 231,
            },
            TLahAccount{
                .Uid = 312,
            },
        };
        NDbPool::TQuery q = h.Fetcher.BuildBbQuery(accs);
        UNIT_ASSERT_STRINGS_EQUAL(q.Query(), "/blackbox");
        UNIT_ASSERT_STRINGS_EQUAL(q.HttpMethod(), "POST");
        UNIT_ASSERT_STRINGS_EQUAL(q.HttpBody(), "attributes=4,136&format=json&method=userinfo&regname&userip=0.0.0.0&uid=123,231,312");
    }

    Y_UNIT_TEST(genJsonWithOrder1) {
        TAccountsInfo info;
        info.insert({"4000849384",
                     R"(😂 Я 'Тестовый" / &<тестер>)",
                     R"(1132/4564654-12313)",
                     false,
                     100500,
                     118});
        info.insert({"4006769293",
                     R"(yandex-team-mm-15040)",
                     R"(0/0-0)",
                     true,
                     1,
                     179});
        info.insert({"7401",
                     R"(yandex-team-mm-15041)",
                     R"(0/0-0)",
                     true,
                     1496749999,
                     188});

        TLahAccounts accounts;
        accounts.insert(TLahAccount{4006769293, 1496740665, 1});
        accounts.insert(TLahAccount{4000849384, 1496740648, 1});

        TString json = TSuggestProcessor::SerializeJson(
            accounts,
            info,
            TSuggestProcessor::ENeed::All,
            {});
        UNIT_ASSERT_STRINGS_EQUAL(R"({"accounts":[{"display_name":"yandex-team-mm-15040","uid":4006769293,"method":null,"avatar":{"default":"0\/0-0","empty":true}},{"display_name":"\uD83D\uDE02 \u042F '\u0422\u0435\u0441\u0442\u043E\u0432\u044B\u0439\" \/ &<\u0442\u0435\u0441\u0442\u0435\u0440>","uid":4000849384,"method":null,"avatar":{"default":"1132\/4564654-12313","empty":false}}]})",
                                  json);

        json = TSuggestProcessor::SerializeJson(
            accounts,
            info,
            TSuggestProcessor::ENeed::One,
            {});
        UNIT_ASSERT_STRINGS_EQUAL(R"({"accounts":[{"display_name":"yandex-team-mm-15040","uid":4006769293,"method":null,"avatar":{"default":"0\/0-0","empty":true}}]})",
                                  json);
    }

    Y_UNIT_TEST(genJsonWithOrder2) {
        TAccountsInfo info;
        info.insert({"4000849384",
                     R"(😂 Я 'Тестовый" / &<тестер>)",
                     R"(1132/4564654-12313)",
                     false,
                     100500,
                     118});
        info.insert({"4006769293",
                     R"(yandex-team-mm-15040)",
                     R"(0/0-0)",
                     true,
                     1496749999,
                     179});
        info.insert({"7401",
                     R"(yandex-team-mm-15041)",
                     R"(0/0-0)",
                     true,
                     132,
                     188});

        TLahAccounts accounts;
        accounts.insert(TLahAccount{4006769293, 1496740665, 1});
        accounts.insert(TLahAccount{4000849384, 1496740648, 1});

        const TString res = R"({"accounts":[{"display_name":"\uD83D\uDE02 \u042F '\u0422\u0435\u0441\u0442\u043E\u0432\u044B\u0439\" \/ &<\u0442\u0435\u0441\u0442\u0435\u0440>","uid":4000849384,"method":null,"avatar":{"default":"1132\/4564654-12313","empty":false}}]})";

        TString json = TSuggestProcessor::SerializeJson(
            accounts,
            info,
            TSuggestProcessor::ENeed::All,
            {});
        UNIT_ASSERT_STRINGS_EQUAL(res, json);

        json = TSuggestProcessor::SerializeJson(
            accounts,
            info,
            TSuggestProcessor::ENeed::One,
            {});
        UNIT_ASSERT_STRINGS_EQUAL(res, json);
    }

    Y_UNIT_TEST(genJsonMethods) {
        TAccountsInfo info;
        info.insert({.Uid = "4000849384"});
        info.insert({.Uid = "4006769293"});
        info.insert({.Uid = "7401"});

        TLahAccounts accounts;
        accounts.insert(TLahAccount{4000849384, 1, 0});
        accounts.insert(TLahAccount{4006769293, 1, 1});
        accounts.insert(TLahAccount{7401, 1, 100500});
        accounts.insert(TLahAccount{7401, 1, 2});

        TString json = TSuggestProcessor::SerializeJson(
            accounts,
            info,
            TSuggestProcessor::ENeed::All,
            TRuntimeContext::TMethodsConfig{
                .Mapping = {
                    {1, "password"},
                    {2, "otp"},
                },
            });
        UNIT_ASSERT_STRINGS_EQUAL(R"({"accounts":[{"display_name":"","uid":7401,"method":"otp","avatar":{"default":"","empty":false}},{"display_name":"","uid":7401,"method":null,"avatar":{"default":"","empty":false}},{"display_name":"","uid":4006769293,"method":"password","avatar":{"default":"","empty":false}},{"display_name":"","uid":4000849384,"method":null,"avatar":{"default":"","empty":false}}]})",
                                  json);
    }
}
