#include <passport/infra/daemons/sezamapi/src/utils/origin.h>
#include <passport/infra/daemons/sezamapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NSezamApi;

Y_UNIT_TEST_SUITE(UtilsTest) {
    static const TOrigin::TDomains TLD =
        {"ru",
         "az",
         "com.am",
         "com.ge",
         "co.il",
         "kg",
         "lv",
         "lt",
         "md",
         "tj",
         "tm",
         "uz",
         "fr",
         "ee",
         "ua",
         "by",
         "kz",
         "com",
         "com.tr"};

    static const TOrigin::TDomains SIDE_DOMAINS = {"kinopoisk.ru", "auto.ru"};

    Y_UNIT_TEST(origin) {
        UNIT_ASSERT_STRINGS_EQUAL(
            R"(^https://(?:[\w_\-.]+\.)?yandex\.(ru|az|com\.am|com\.ge|co\.il|kg|lv|lt|md|tj|tm|uz|fr|ee|ua|by|kz|com|com\.tr)$)",
            TOrigin::RegexForYandexTld(TOrigin::EPolicy::Strict, "yandex", TLD));
        TOriginPtr tld = TOrigin::CreateForYandexTld(TOrigin::EPolicy::Strict, "yandex", TLD);

        UNIT_ASSERT(!tld->Check(""));

        UNIT_ASSERT(tld->Check("https://yandex.ru"));
        UNIT_ASSERT(tld->Check("https://yandex.ua"));
        UNIT_ASSERT(tld->Check("https://yandex.az"));
        UNIT_ASSERT(tld->Check("https://yandex.com"));
        UNIT_ASSERT(tld->Check("https://yandex.com.ge"));
        UNIT_ASSERT(tld->Check("https://yandex.com.tr"));
        UNIT_ASSERT(tld->Check("https://yandex.co.il"));
        UNIT_ASSERT(tld->Check("https://yandex.kg"));
        UNIT_ASSERT(tld->Check("https://yandex.lv"));
        UNIT_ASSERT(tld->Check("https://yandex.lt"));
        UNIT_ASSERT(tld->Check("https://yandex.md"));
        UNIT_ASSERT(tld->Check("https://yandex.tj"));
        UNIT_ASSERT(tld->Check("https://yandex.tm"));
        UNIT_ASSERT(tld->Check("https://yandex.uz"));
        UNIT_ASSERT(tld->Check("https://yandex.fr"));
        UNIT_ASSERT(tld->Check("https://yandex.ee"));
        UNIT_ASSERT(tld->Check("https://yandex.by"));
        UNIT_ASSERT(tld->Check("https://yandex.kz"));

        UNIT_ASSERT(tld->Check("https://.mail.yandex.ru"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.ua"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.az"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.com"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.com.ge"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.com.tr"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.co.il"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.kg"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.lv"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.lt"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.md"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.tj"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.tm"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.uz"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.fr"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.ee"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.by"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex.kz"));

        UNIT_ASSERT(tld->Check("https://05.yandex.kz"));

        UNIT_ASSERT(!tld->Check("http://yandex.ru"));
        UNIT_ASSERT(!tld->Check("https://yandex_ru"));
        UNIT_ASSERT(!tld->Check("https://evil-yandex.ru"));
        UNIT_ASSERT(!tld->Check("https://yandex.ru.evil.com"));
        UNIT_ASSERT(!tld->Check("https://yandex.com.evil.com"));
        UNIT_ASSERT(!tld->Check("https://yandex.ru:8080"));

        UNIT_ASSERT(!tld->Check("https://kinopoisk.ru"));

        UNIT_ASSERT(!tld->Check("https://yandex-team.ru"));

        UNIT_ASSERT_VALUES_EQUAL("yandex.md", *tld->Check("https://.mail.yandex.md"));
        UNIT_ASSERT_VALUES_EQUAL("yandex.ru", *tld->Check("https://yandex.ru"));
    }

    Y_UNIT_TEST(origin_yateam) {
        TOrigin::TDomains tldCfg = {"ru", "com"};
        UNIT_ASSERT_STRINGS_EQUAL(
            R"(^https://(?:[\w_\-.]+\.)?yandex-team\.(ru|com)$)",
            TOrigin::RegexForYandexTld(TOrigin::EPolicy::Strict, "yandex-team", tldCfg));
        UNIT_ASSERT_STRINGS_EQUAL(
            R"(^https?://(?:[\w_\-.]+\.)?yandex-team\.(ru|com)(:[\d]{1,5})?$)",
            TOrigin::RegexForYandexTld(TOrigin::EPolicy::Relaxed, "yandex-team", tldCfg));
        TOriginPtr tld = TOrigin::CreateForYandexTld(TOrigin::EPolicy::Strict, "yandex-team", tldCfg);
        TOriginPtr relaxedTld = TOrigin::CreateForYandexTld(TOrigin::EPolicy::Relaxed, "yandex-team", tldCfg);

        UNIT_ASSERT(!tld->Check(""));
        UNIT_ASSERT(!relaxedTld->Check(""));

        UNIT_ASSERT(tld->Check("https://yandex-team.ru"));
        UNIT_ASSERT(tld->Check("https://yandex-team.com"));
        UNIT_ASSERT(relaxedTld->Check("https://yandex-team.ru"));
        UNIT_ASSERT(relaxedTld->Check("https://yandex-team.com"));

        UNIT_ASSERT(tld->Check("https://.mail.yandex-team.ru"));
        UNIT_ASSERT(tld->Check("https://.mail.yandex-team.com"));
        UNIT_ASSERT(relaxedTld->Check("https://.mail.yandex-team.ru"));
        UNIT_ASSERT(relaxedTld->Check("https://.mail.yandex-team.com"));

        UNIT_ASSERT(!tld->Check("http://yandex-team.ru"));
        UNIT_ASSERT(!tld->Check("https://yandex-team_ru"));
        UNIT_ASSERT(!tld->Check("https://evil-yandex-team.ru"));
        UNIT_ASSERT(!tld->Check("https://yandex-team.ru.evil.com"));
        UNIT_ASSERT(!tld->Check("https://yandex-team.com.evil.com"));
        UNIT_ASSERT(!tld->Check("https://yandex-team.ru:8080"));
        UNIT_ASSERT(relaxedTld->Check("http://yandex-team.ru"));
        UNIT_ASSERT(!relaxedTld->Check("https://yandex-team_ru"));
        UNIT_ASSERT(!relaxedTld->Check("https://evil-yandex-team.ru"));
        UNIT_ASSERT(!relaxedTld->Check("https://yandex-team.ru.evil.com"));
        UNIT_ASSERT(!relaxedTld->Check("https://yandex-team.com.evil.com"));
        UNIT_ASSERT(relaxedTld->Check("https://yandex-team.ru:8080"));

        UNIT_ASSERT(!tld->Check("https://kinopoisk.ru"));
        UNIT_ASSERT(!relaxedTld->Check("https://kinopoisk.ru"));

        UNIT_ASSERT(!tld->Check("https://yandex.ru"));
        UNIT_ASSERT(!relaxedTld->Check("https://yandex.ru"));

        UNIT_ASSERT_VALUES_EQUAL("yandex-team.com", *tld->Check("https://.mail.yandex-team.com"));
        UNIT_ASSERT_VALUES_EQUAL("yandex-team.ru", *tld->Check("https://yandex-team.ru"));
        UNIT_ASSERT_VALUES_EQUAL("yandex-team.com", *relaxedTld->Check("https://.mail.yandex-team.com"));
        UNIT_ASSERT_VALUES_EQUAL("yandex-team.ru", *relaxedTld->Check("https://yandex-team.ru"));
    }

    Y_UNIT_TEST(createForSideDomains) {
        UNIT_ASSERT(!TOrigin::CreateForSideDomains(TOrigin::EPolicy::Strict, {}));
    }

    Y_UNIT_TEST(kinopoisk) {
        UNIT_ASSERT_STRINGS_EQUAL(
            R"(^https://(?:[\w_\-.]+\.)?(kinopoisk\.ru|auto\.ru)$)",
            TOrigin::RegexForSideDomains(TOrigin::EPolicy::Strict, SIDE_DOMAINS));
        UNIT_ASSERT_STRINGS_EQUAL(
            R"(^https?://(?:[\w_\-.]+\.)?(kinopoisk\.ru|auto\.ru)(:[\d]{1,5})?$)",
            TOrigin::RegexForSideDomains(TOrigin::EPolicy::Relaxed, SIDE_DOMAINS));

        TOriginPtr relaxed = TOrigin::CreateForSideDomains(TOrigin::EPolicy::Relaxed, SIDE_DOMAINS);
        TOriginPtr strict = TOrigin::CreateForSideDomains(TOrigin::EPolicy::Strict, SIDE_DOMAINS);

        auto t = [](const TOriginPtr& o, const TString& dom) {
            UNIT_ASSERT(o->Check(NUtils::ReplaceAny("https://*", "*", dom)));
            UNIT_ASSERT_VALUES_EQUAL(dom, *o->Check(NUtils::ReplaceAny("https://*", "*", dom)));
            UNIT_ASSERT(o->Check(NUtils::ReplaceAny("https://www.*", "*", dom)));
            UNIT_ASSERT_VALUES_EQUAL(dom, *o->Check(NUtils::ReplaceAny("https://www.*", "*", dom)));
            UNIT_ASSERT(!o->Check(NUtils::ReplaceAny("https://www.*.evil.com", "*", dom)));
            UNIT_ASSERT(!o->Check(NUtils::ReplaceAny("*", "*", dom)));
            UNIT_ASSERT(!o->Check(NUtils::ReplaceAny("https://www.*.evil.com:3000", "*", dom)));
            UNIT_ASSERT(!o->Check(NUtils::ReplaceAny("*:3000", "*", dom)));
        };

        t(relaxed, "kinopoisk.ru");
        t(relaxed, "auto.ru");
        t(strict, "kinopoisk.ru");
        t(strict, "auto.ru");

        std::vector<TString> ok = {
            "https://*:1",
            "https://*:3000",
            "https://*:99999",
            "http://*",
            "http://*:1",
            "http://*:3000",
            "http://*:99999",
        };
        for (const TString& s : ok) {
            UNIT_ASSERT(relaxed->Check(NUtils::ReplaceAny(s, "*", "kinopoisk.ru")));
            UNIT_ASSERT(!strict->Check(NUtils::ReplaceAny(s, "*", "kinopoisk.ru")));
            UNIT_ASSERT_VALUES_EQUAL("kinopoisk.ru", *relaxed->Check(NUtils::ReplaceAny(s, "*", "kinopoisk.ru")));

            UNIT_ASSERT(relaxed->Check(NUtils::ReplaceAny(s, "*", "auto.ru")));
            UNIT_ASSERT(!strict->Check(NUtils::ReplaceAny(s, "*", "auto.ru")));
            UNIT_ASSERT_VALUES_EQUAL("auto.ru", *relaxed->Check(NUtils::ReplaceAny(s, "*", "auto.ru")));
        }

        std::vector<TString> fail = {
            "*:1",
            "*:3000",
            "*:99999",
        };
        for (const TString& s : fail) {
            UNIT_ASSERT(!relaxed->Check(NUtils::ReplaceAny(s, "*", "kinopoisk.ru")));
            UNIT_ASSERT(!strict->Check(NUtils::ReplaceAny(s, "*", "kinopoisk.ru")));
        }

        UNIT_ASSERT(!relaxed->Check("https://yandex.ru"));
        UNIT_ASSERT(!strict->Check("https://yandex.ru"));
    }

    Y_UNIT_TEST(getPostponeDelay) {
        UNIT_ASSERT_VALUES_EQUAL(15, TUtils::GetPostponeDelay("asd", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(15, TUtils::GetPostponeDelay("1asd", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(15, TUtils::GetPostponeDelay("-70", 10, 100, 15));

        UNIT_ASSERT_VALUES_EQUAL(15, TUtils::GetPostponeDelay("0", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(15, TUtils::GetPostponeDelay("700", 10, 100, 15));

        UNIT_ASSERT_VALUES_EQUAL(10, TUtils::GetPostponeDelay("10", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(20, TUtils::GetPostponeDelay("20", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(70, TUtils::GetPostponeDelay("70", 10, 100, 15));
        UNIT_ASSERT_VALUES_EQUAL(100, TUtils::GetPostponeDelay("100", 10, 100, 15));
    }

    Y_UNIT_TEST(getTsFromCookie) {
        UNIT_ASSERT_VALUES_EQUAL(10, TUtils::GetTsFromCookie("10"));
        UNIT_ASSERT_VALUES_EQUAL(0, TUtils::GetTsFromCookie("abc"));
        UNIT_ASSERT_VALUES_EQUAL(0, TUtils::GetTsFromCookie("999999999999999999999999"));
        UNIT_ASSERT_VALUES_EQUAL(0, TUtils::GetTsFromCookie(""));
    }

    Y_UNIT_TEST(PrepareYpCookieDomain) {
        UNIT_ASSERT_VALUES_EQUAL("passport.yandex.ru", TUtils::PrepareYpCookieDomain("new.api.passport.yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("yandex.ru", TUtils::PrepareYpCookieDomain("api.passport.yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("yandex.com.tr", TUtils::PrepareYpCookieDomain("api.passport.yandex.com.tr"));
        UNIT_ASSERT_VALUES_EQUAL("", TUtils::PrepareYpCookieDomain("api.passport"));
        UNIT_ASSERT_VALUES_EQUAL(".", TUtils::PrepareYpCookieDomain("..."));
    }
}
