package main

import (
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strconv"
	"strings"
	"time"
)

const lrandoms = `7223;aVssQNNa6b1yMC1YkGmmHyUjhGDXUKj5oVmR2I9CROZPH57q5dV61AapGSnoPW0a;1478120402
7221;zmSKyM4C1yPxikUCeFWcbMHi25VciraFFWVTbh8NTy1dFxRukUlyU3qEOcHKPcGw;1478120402
7219;8DmWWlYsDSJZbYRn0MIqS5i5N2rxcp1Fjav9GGWFKDqR6XsrGOZDDy56Bj0luyuq;1478120402
7217;jXxyeMXx9FYsUpjSt9avW9drKP6wUwHC6Q2l2gYz0u5nWzkbVOBlDYiRif24WWm2;1478120402
7215;Gtg0xsNeKGI6vfld6tsg2v1nL8t4PXVFioTPBTC0XAsfecKHCxjZVesoJuV2pN5A;1478120402
7213;1R5oQApRMiFxo5G6paqQQLCHcr75eBEyCgsbx8pqJOR8wzS64zgyR87xNUBuY7CU;1478120402
7211;D1uKcRhxyNsusQP72Zb1ljdqbquY9LI5R3ofvKPFbs9cUETOUGo3Ov2oEhRRaAVI;1478120402
7209;UxdR5zAeAhFHID2TXN40ywRDf5MGWc0YVMsx1XrLsy8BrrjMG1BaGGNrOoKrji5Y;1478034002
7207;hVAkQUf9hBNovlbJ5Fdg8Qpd5ldry9eyIRIyCyMfwNuv7BpbXANiWc41byjU2YPZ;1478034002
7205;CF7bSx3Fwx4Of0dBlSGCORDwKpfXvjBAPsiz1ssixPcvwGxRnHnhtdbj6J10SRho;1478034002
7203;c0N90qLV8pCvVMLoYCuYDkO6J8q8bPPtYeZEzgsYwJ2gVgkzXGOVN1qJdUlEqGCJ;1478034002
7201;0cy9AD4pR6zTOtrtbaLNYyvUduh6TWirIKU0590nn4K3beLPd0qFsqzER5urv1jJ;1478034002
7199;G3kIIaO4ISAHCbODZ8S1iu8ieQ3ZrI0iZxTRVkCKaOevI31b7xrRwKn9xuIZhdwh;1478034002
7197;qxYBOAWCkCRb41lmtNyrKpgL0JKGjLwpHxlnRVyuyRQWpXELpUsTn8f0YTE1pxSu;1478034002
7195;xqVqY0odzgFV0Fmpi4U8xT62dqnMp4smq5TanEUvCEKmhqi635WvjtQyNxPaLQBa;1478034001
7193;KJLB49JWhb9c3q0ZbvXvk1dr6GPiUBPQNbP7gBwvzW2kXNdTVcK9CDZGoyfR39EN;1477947602
7191;XplhzqDUtw8Gkf2jUJMChBrmNdBmmA8tm0wU2GZx0sQs8ZsP8TteOaCT3eVgXXVV;1477947602
7189;KJl5WQreh9U5n3MsHYo5OEcjlCvUjBU7iNouRrG6NAvgn8DtDNkZd7krkHZ0cSQc;1477947602
7187;I75znFUHgiHT7ntVn2OGnpYXBLuLZEPRg9fK6yU3Z79NmNoxhLno2psuU12Uncjn;1477947602
7185;wnBVqpOtIx5tP7f941qE2wUqxwL3G3kM5fAi2KU75o5Pl3oSnZBpUVP6WribWXN4;1477947602
7183;FOeOdRjr3Nk9s3hfB6vmOHxRHBsCbuSzS2edo3zquibfUmsWqvnqIx39yXit23Vc;1477947602
7181;4v3LprBIHkA7zkFYdW2CIcDNkEsMtBXcKHIhbqjc3uBSk0awlue4BIdonzhhQoIP;1477947602
7179;k5vHGEtFDGXAQ25xe46nMZmhhz3DdTB1pqarXjMwEziaU49iyGQTJsirOIC1Y5PL;1477947602
`

func mainImpl() error {
	if len(os.Args) < 4 {
		return errors.New("usage: nginx_mock <port> <bb_port> <sezamapi_port>")
	}

	nginxPort, err := strconv.Atoi(os.Args[1])
	if err != nil {
		return err
	}

	bbPort, err := strconv.Atoi(os.Args[2])
	if err != nil {
		return err
	}

	sezamapiPort, err := strconv.Atoi(os.Args[3])
	if err != nil {
		return err
	}

	logger := log.New(os.Stdout, "nginx_mock: ", log.LstdFlags)
	logger.Printf("Starting nginx mock server at port: %d. Bb port: %d. Sezamapi port: %d\n", nginxPort, bbPort, sezamapiPort)

	defaultHandler := func(w http.ResponseWriter, r *http.Request) {
		if r.URL.Path == "/lrandoms.txt" {
			_, _ = fmt.Fprint(w, lrandoms)
			return
		}

		backendPort := sezamapiPort
		path := r.URL.Path
		if !strings.Contains(r.Host, "api.my-passport.yandex.ru") {
			backendPort = bbPort
			if path == "/ping" {
				path = "/blackbox/nagios"
			}
		}

		backend := fmt.Sprintf("http://localhost:%d%s?%s", backendPort, path, r.URL.RawQuery)
		req, err := http.NewRequest(r.Method, backend, r.Body)
		if err != nil {
			panic(err)
		}
		for k, v := range r.Header {
			req.Header.Set(k, v[0])
		}
		for _, c := range r.Cookies() {
			req.AddCookie(c)
		}
		req.Host = r.Host

		client := http.Client{}
		resp, err := client.Do(req)
		if err != nil {
			w.WriteHeader(http.StatusServiceUnavailable)
			_, _ = w.Write([]byte(err.Error()))
		}

		defer func() { _ = resp.Body.Close() }()
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			panic(err)
		}

		for k, v := range resp.Header {
			w.Header().Set(k, v[0])
		}
		w.WriteHeader(resp.StatusCode)
		_, _ = w.Write(body)

		logger.Printf("in: %s (Host: %s, Origin: %s). proxied to: %s. got (%d): %s",
			r.URL.Path,
			r.Host,
			r.Header.Get("Origin"),
			backend,
			resp.StatusCode,
			string(body))
	}

	server := &http.Server{
		Addr:         ":" + os.Args[1],
		Handler:      http.HandlerFunc(defaultHandler),
		ErrorLog:     logger,
		ReadTimeout:  5 * time.Second,
		WriteTimeout: 10 * time.Second,
		IdleTimeout:  15 * time.Second,
	}

	if err := server.ListenAndServe(); err != http.ErrServerClosed {
		return err
	}

	return nil
}

func main() {
	if err := mainImpl(); err != nil {
		log.Fatalf("Fatal error: %s", err.Error())
	}
}
