package auth

import (
	"crypto/rand"
	"encoding/base64"
	"fmt"
	"io/ioutil"
	"net"
	"os"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/config"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func CreateHTTPClient(cfg *config.Config) *resty.Client {
	client := resty.New().
		SetHostURL(cfg.ShooterURL).
		SetTimeout(45 * time.Second).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	return client
}

func CreateRequestWithClient(cfg *config.Config, client *resty.Client) *resty.Request {
	timestamp := fmt.Sprintf("%d", time.Now().Unix())

	res := client.R()
	res.SetHeader(
		"Authorization",
		fmt.Sprintf("%s %s %s",
			cfg.Username,
			timestamp,
			base64.StdEncoding.EncodeToString(createSignature(cfg, timestamp)),
		),
	)

	return res
}

func CreateRequest(cfg *config.Config) *resty.Request {
	return CreateRequestWithClient(cfg, CreateHTTPClient(cfg))
}

func createSignature(cfg *config.Config, toSign string) []byte {
	sign, err := createWithAgent(toSign)
	if err == nil {
		return sign
	}
	logger.Log().Warnf("failed to create sign with ssh agent: %s", err)

	sign, err = createWithPrivateKey(cfg, toSign)
	if err == nil {
		return sign
	}
	logger.Log().Warnf("failed to create sign with private key: %s", err)

	return nil
}

func createWithAgent(toSign string) ([]byte, error) {
	sshAgentConn, err := net.Dial("unix", os.Getenv("SSH_AUTH_SOCK"))
	if err != nil {
		return nil, xerrors.Errorf("failed to connect to ssh agent: %s", err)
	}

	ag := agent.NewClient(sshAgentConn)
	signers, err := ag.Signers()
	if err != nil {
		return nil, xerrors.Errorf("failed to get signers: %s", err)
	}

	for _, signer := range signers {
		if strings.Contains(signer.PublicKey().Type(), "cert") {
			logger.Log().Debugf("ignore certs from skotty")
			continue
		}

		sign, err := signer.Sign(rand.Reader, []byte(toSign))
		if err != nil {
			logger.Log().Warnf("failed to sign with ssh agent: %s", err)
			continue
		}

		return ssh.Marshal(sign), nil
	}

	return nil, xerrors.Errorf("failed to make any sign with agent: %s", err)
}

func createWithPrivateKey(cfg *config.Config, toSign string) ([]byte, error) {
	if cfg.SSHPrivateKey == "" {
		return nil, xerrors.Errorf("no private key for ssh")
	}

	pkey, err := ioutil.ReadFile(cfg.SSHPrivateKey)
	if err != nil {
		return nil, xerrors.Errorf("failed to read file: %s", err)
	}

	signer, err := ssh.ParsePrivateKey(pkey)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse private key: %s", err)
	}

	sign, err := signer.Sign(rand.Reader, []byte(toSign))
	if err != nil {
		return nil, xerrors.Errorf("failed to sign: %s", err)
	}

	return ssh.Marshal(sign), nil
}
