package ammo

import (
	"encoding/json"
	"fmt"
	"os"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/auth"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/clitypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func cmdCreate(cfg *config.Config) *cobra.Command {
	var duration uint32
	var hosts string

	res := &cobra.Command{
		Use:   "create",
		Short: "Create new ammo pack",
		RunE: func(cmd *cobra.Command, args []string) error {
			client := auth.CreateHTTPClient(cfg)

			path := fmt.Sprintf(
				"/cli/ammo/create?hosts=%s&duration=%d",
				hosts,
				duration,
			)

			code, resp, err := misc.GetResponse(auth.CreateRequestWithClient(cfg, client).Post(path))
			if err != nil || code != 200 {
				return xerrors.Errorf("failed to create ammo: %w. code=%d.\n%s", err, code, string(resp))
			}

			var createRes clitypes.AmmoCreateResult
			if err := json.Unmarshal(resp, &createRes); err != nil {
				return xerrors.Errorf("failed to parse creating ammo result: %w. \n%s", err, string(resp))
			}
			logger.Log().Infof("Waiting for new ammo: %s", createRes.ID)

			for {
				time.Sleep(10 * time.Second)

				code, resp, err = misc.GetResponse(auth.CreateRequestWithClient(cfg, client).Get("/cli/ammo/list"))
				if err != nil || code != 200 {
					return xerrors.Errorf("failed to get ammo list: %w. code=%d.\n%s", err, code, string(resp))
				}

				var list clitypes.ListResult
				if err := json.Unmarshal(resp, &list); err != nil {
					return xerrors.Errorf("failed to parse ammo list: %w.\n%s", err, string(resp))
				}

				info, ok := list[createRes.ID]
				if !ok {
					return xerrors.Errorf("failed to find ammo id in shooter response; %s: \n%s", createRes.ID, string(resp))
				}

				if info.Status == clitypes.AmmoInProgress.String() {
					logger.Log().Debugf("Ammo is in progress now")
					continue
				}
				if info.Status == clitypes.AmmoInvalid.String() {
					return xerrors.Errorf("failed to create ammo: \n%s", string(resp))
				}

				break
			}
			_, _ = os.Stdout.Write(misc.Prettify(resp))

			return nil
		},
	}

	flags := res.Flags()
	flags.Uint32Var(&duration, "duration", 0, "duration of sampling")
	flags.StringVar(&hosts, "hosts", "", "blackbox hosts")

	misc.MarkFlagRequired(flags, "duration", "hosts")

	return res
}
