package perf

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/auth"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func cmdGet(cfg *config.Config) *cobra.Command {
	var noFlame bool

	res := &cobra.Command{
		Use:   "get",
		Short: "Fetch ready perf",
		Long:  "Fetch ready perf. Result will be saved in ~/.shg-cli/ with flame graphs",
		RunE: func(cmd *cobra.Command, args []string) error {
			_, err := getImpl(cfg, auth.CreateHTTPClient(cfg), noFlame)
			return err
		},
	}

	flags := res.Flags()
	flags.BoolVar(&noFlame, "no-flamegraph", false, "do not create FlameGraph")

	return res
}

func getImpl(cfg *config.Config, client *resty.Client, noFlame bool) (bool, error) {
	perf, err := getPerfResponse(cfg, client)
	if err != nil {
		return false, err
	}
	if perf == nil {
		return true, nil
	}

	data, err := getRawPerf(perf)
	if err != nil {
		return false, err
	}

	filename := cfg.HomeDir + "/perf.script.data"
	if err := ioutil.WriteFile(filename, data, os.FileMode(0666)); err != nil {
		return false, xerrors.Errorf("failed to write perf: %w", err)
	}
	logger.Log().Infof("perf saved: %s", filename)

	if !noFlame {
		if err := initFlameGraph(cfg); err != nil {
			return false, xerrors.Errorf("failed to init flame graph: %w", err)
		}

		if err := genFlameGraph(cfg, filename); err != nil {
			return false, xerrors.Errorf("failed to generate flame graph: %w", err)
		}
	}

	return false, nil
}

func getPerfResponse(cfg *config.Config, client *resty.Client) (*stateviewertypes.Perf, error) {
	time.Sleep(10 * time.Second)
	code, resp, err := misc.GetResponse(auth.CreateRequestWithClient(cfg, client).Get("/cli/state/perf"))
	if err != nil {
		return nil, xerrors.Errorf("failed to fetch perf: %w. code=%d.\n%s", err, code, string(resp))
	}

	if code == 400 { // TODO
		logger.Log().Debug("perf is not ready yet")
		return nil, nil
	}
	if code != 200 {
		return nil, xerrors.Errorf("failed to fetch perf. code=%d.\n%s", code, string(resp))
	}

	var perf stateviewertypes.Perf
	_ = json.Unmarshal(resp, &perf)
	return &perf, nil
}

func getRawPerf(perf *stateviewertypes.Perf) ([]byte, error) {
	data, err := base64.RawURLEncoding.DecodeString(perf.Output)
	if err != nil {
		return nil, xerrors.Errorf("failed to decode base64url with perf: %s: %w", perf.Output, err)
	}
	data, err = misc.Decompress(data)
	if err != nil {
		return nil, xerrors.Errorf("failed to decompress perf: %s: %w", perf.Output, err)
	}

	return data, nil
}

func initFlameGraph(cfg *config.Config) error {
	if _, err := os.Stat(flameDir(cfg)); err == nil {
		return nil // already inited
	}

	cmd := fmt.Sprintf("git clone https://github.com/brendangregg/FlameGraph -q %s", flameDir(cfg))
	if _, err := misc.RunCmd(cmd); err != nil {
		return xerrors.Errorf("failed to clone flame graph scripts: %w", err)
	}

	return nil
}

func genFlameGraph(cfg *config.Config, perfData string) error {
	tmpFile := fmt.Sprintf("%s/out.perf-folded", cfg.HomeDir)
	outFile := fmt.Sprintf("%s/perf.svg", cfg.HomeDir)
	outReversedFile := fmt.Sprintf("%s/perf-reversed.svg", cfg.HomeDir)

	cmd := fmt.Sprintf(
		"cat %s | __flamedir__/stackcollapse-perf.pl > __tmpfile__ &&"+
			" __flamedir__/flamegraph.pl --width 2048 __tmpfile__ > %s &&"+
			" __flamedir__/flamegraph.pl --reverse --width 32768 __tmpfile__ > %s",
		perfData,
		outFile,
		outReversedFile,
	)
	cmd = strings.Replace(cmd, "__flamedir__", flameDir(cfg), -1)
	cmd = strings.Replace(cmd, "__tmpfile__", tmpFile, -1)

	if _, err := misc.RunCmd(cmd); err != nil {
		return err
	}

	if err := os.Remove(tmpFile); err != nil {
		logger.Log().Warnf("Failed to remove temp file: %s", tmpFile)
	}

	logger.Log().Infof("Flame graph saved: file://%s", outFile)
	logger.Log().Infof("Flame reverse graph saved: file://%s", outReversedFile)

	return nil
}

func flameDir(cfg *config.Config) string {
	return cfg.HomeDir + "/flame"
}
