package version

import (
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/auth"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/cli/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func cmdInstall(cfg *config.Config) *cobra.Command {
	res := &cobra.Command{
		Use:     "install",
		Short:   "Install debian package with specified version",
		Long:    "Install debian package with specified version. Allowed list you can see in `shg-cli version get`",
		Example: "shg-cli version install yandex-passport-blackbox=1.2.3",
		RunE: func(cmd *cobra.Command, args []string) error {
			if len(args) != 1 {
				return xerrors.Errorf("Too many args: %s", args)
			}
			fields := strings.Split(args[0], "=")
			pkg := fields[0]
			taskVer := fields[1]

			client := auth.CreateHTTPClient(cfg)

			path := fmt.Sprintf("/cli/version?package=%s&version=%s", pkg, taskVer)
			code, resp, err := misc.GetResponse(auth.CreateRequestWithClient(cfg, client).Post(path))
			if err != nil || code != 200 {
				return xerrors.Errorf("failed to post task: %w. code=%d.\n%s", err, code, string(resp))
			}

			logger.Log().Infof("Installing command was sent. Waiting...")
			for {
				time.Sleep(15 * time.Second)

				code, resp, err := misc.GetResponse(auth.CreateRequestWithClient(cfg, client).Get("/cli/version"))
				if err != nil || code != 200 {
					return xerrors.Errorf("failed to get versions: %w. code=%d.\n%s", err, code, string(resp))
				}

				var versions stateviewertypes.Versions
				_ = json.Unmarshal(resp, &versions)

				actualVer, ok := versions[pkg]
				if !ok {
					logger.Log().Warnf("Missing package %s in response: %s", pkg, resp)
					continue
				}
				if actualVer != taskVer {
					logger.Log().Debugf("Actual version of %s is %s. Waiting for %s", pkg, actualVer, taskVer)
					continue
				}

				_, _ = os.Stdout.Write(resp)
				return nil
			}
		},
	}

	return res
}
