package prospector

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type cfgShooter struct {
	URL string `json:"url"`
}

type cfgAmmo struct {
	Dir           string `json:"dir"`
	GorPath       string `json:"gor_path"`
	ListenPort    uint16 `json:"listen_port"`
	LocalUser     string `json:"local_user"`
	FileSizeLimit uint64 `json:"file_size_limit"`
}

type cfgScp struct {
	RemoteUser     string `json:"remote_user"`
	Host           string `json:"host"`
	PrivateKeyPath string `json:"private_key_path"`
	RemoteDir      string `json:"remote_dir"`
	BandwidthKbits uint64 `json:"bandwidth_kbits"`
}

type Config struct {
	Shooter cfgShooter `json:"shooter"`
	Ammo    cfgAmmo    `json:"ammo"`
	Scp     cfgScp     `json:"scp"`
}

type Prospector struct {
	cfg      Config
	stop     chan bool
	meta     Meta
	shooter  *resty.Client
	hostname string
}

type Meta struct {
	ID string `json:"last_task_id"`
}

const (
	metaFileName = ".meta.json"
)

func NewProspector(cfg Config) (*Prospector, error) {
	cr := cmdRunner{}

	shooter := resty.New().
		SetHostURL(cfg.Shooter.URL).
		SetTimeout(5 * time.Second).
		SetRedirectPolicy(resty.NoRedirectPolicy())
	if err := checkShooterURL(shooter); err != nil {
		return nil, err
	}

	err := checkCfg(cfg, &cr)
	if err != nil {
		return nil, xerrors.Errorf("Failed on checking config: %w", err)
	}

	hostname, err := cr.Run("hostname -f")
	if err != nil {
		return nil, xerrors.Errorf("failed to get hostname: %s", err)
	}

	res := &Prospector{
		cfg:      cfg,
		stop:     make(chan bool),
		meta:     readMeta(cfg.Ammo.Dir),
		shooter:  shooter,
		hostname: string(hostname[:len(hostname)-1]),
	}
	logger.Log().Debugf("Detected hostname: '%s'", res.hostname)

	go func() {
		heartbeat := time.NewTicker(5 * time.Second)

		for {
			select {
			case <-res.stop:
				logger.Log().Info("Routine: quitting")
				return

			case <-heartbeat.C:
				if err := res.routine(); err != nil {
					logger.Log().Warnf("Routine: error: %s", err)
				}
			}
		}
	}()

	return res, nil
}

func (p *Prospector) Stop() {
	close(p.stop)
}

func checkCfg(cfg Config, runner CmdRunner) error {
	if err := checkAmmoDir(cfg.Ammo.Dir); err != nil {
		return err
	}
	if err := checkGorPath(cfg.Ammo.GorPath, runner); err != nil {
		return err
	}

	return nil
}

func checkAmmoDir(dir string) error {
	st, err := os.Stat(dir)
	if err != nil {
		return xerrors.Errorf("failed to stat dir for ammo: %s: %w", dir, err)
	}
	if !st.IsDir() {
		return xerrors.Errorf("Path for ammo is not dir: %s", dir)
	}

	fileName := dir + "/.dummy"
	file, err := os.OpenFile(fileName, os.O_RDWR|os.O_CREATE, 0666)
	if err != nil {
		return xerrors.Errorf("failed to check permissions: %s: %w", dir, err)
	}
	_ = file.Close()
	_ = os.Remove(fileName)

	return nil
}

func checkGorPath(path string, runner CmdRunner) error {
	cmd := fmt.Sprintf("sudo %s --input-raw :1 -exit-after 1s -output-null", path)
	_, err := runner.Run(cmd)
	return err
}

func checkShooterURL(httpc *resty.Client) error {
	resp, err := httpc.R().
		SetContext(context.Background()).
		Get("/ping")
	if err != nil {
		return xerrors.Errorf("failed to ping shooter: %w", err)
	}

	if code := resp.StatusCode(); code != 200 {
		return xerrors.Errorf("failed to ping shooter: code %d: %s", code, string(resp.Body()))
	}

	return nil
}

func readMeta(dir string) Meta {
	fileName := fmt.Sprintf("%s/%s", dir, metaFileName)
	body, err := ioutil.ReadFile(fileName)
	if err != nil {
		logger.Log().Warnf("failed to read meta for ammo: %s: %s", dir, err)
		return Meta{}
	}

	var res Meta
	if err := json.Unmarshal(body, &res); err != nil {
		logger.Log().Warnf("failed to parse meta for ammo: %s: %s", dir, err)
		return Meta{}
	}

	return res
}
