package prospector

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/prospectortypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func (p *Prospector) routine() error {
	hg := httpGetter{p.shooter}
	task, err := getTask(p.cfg.Ammo.Dir, p.hostname, p.meta.ID, &hg)
	if err != nil || task == nil {
		return err
	}

	logger.Log().Infof("Got task %s with duration: %d seconds", task.ID, task.Duration)
	p.meta.ID = task.ID
	if err := writeMetaToDisk(p.cfg.Ammo.Dir, p.meta); err != nil {
		return xerrors.Errorf("failed to save meta to disk: %w", err)
	}

	cr := cmdRunner{}
	if err := runTask(&p.cfg, p.hostname, task, &cr); err != nil {
		if er := trySendError(&p.cfg, p.hostname, err, task.ID, &cr); er != nil {
			logger.Log().Warnf("failed to send error '%s' because of: %s", err, er)
		}
		return xerrors.Errorf("failed to exec task from shooter: %s: %w", task.ID, err)
	}

	return nil
}

func getTask(ammoDir string, hostname string, prevTaskID string, hg HTTPGetter) (*prospectortypes.Task, error) {
	if err := cleanupFiles(ammoDir); err != nil {
		return nil, xerrors.Errorf("failed to cleanup files: %w", err)
	}

	code, body, err := hg.Get(fmt.Sprintf("/prospector/task?host=%s", hostname))
	if err != nil {
		return nil, xerrors.Errorf("failed to get task from shooter: %w", err)
	}

	task, err := parseTaskResponse(code, body)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse task from shooter: %w", err)
	}

	if task.ID == "" || task.ID == prevTaskID {
		return nil, nil
	}

	return task, nil
}

func parseTaskResponse(code int, body []byte) (*prospectortypes.Task, error) {
	if code != 200 {
		return nil, xerrors.Errorf("failed to get task from shooter: code %d: %s", code, string(body))
	}

	var res prospectortypes.Task
	if err := json.Unmarshal(body, &res); err != nil {
		return nil, xerrors.Errorf("failed to parse task from shooter: code %d: %s", code, string(body))
	}

	return &res, nil
}

func cleanupFiles(dir string) error {
	files, err := ioutil.ReadDir(dir)
	if err != nil {
		return xerrors.Errorf("failed to read dir with ammo ( %s ): %w", dir, err)
	}

	for _, d := range files {
		if d.Name() == metaFileName {
			continue
		}

		f := fmt.Sprintf("%s/%s", dir, d.Name())
		if err := os.RemoveAll(f); err != nil {
			return xerrors.Errorf("failed to remove file ( %s ): %w", f, err)
		}
	}

	return nil
}

func runTask(cfg *Config, hostname string, task *prospectortypes.Task, runner CmdRunner) error {
	born := uint64(time.Now().Unix())

	logger.Log().Debugf("Task %s: dumping: start", task.ID)
	if err := dumpRequests(&cfg.Ammo, task.Duration, hostname, runner); err != nil {
		return xerrors.Errorf("failed to dump requests: %s", err)
	}
	logger.Log().Debugf("Task %s: dumping: finished", task.ID)

	logger.Log().Debugf("Task %s: correcting permissions: start", task.ID)
	if err := correctPermissions(&cfg.Ammo, hostname, runner); err != nil {
		return xerrors.Errorf("failed to correcting permissions: %s", err)
	}
	logger.Log().Debugf("Task %s: correcting permissions: finished", task.ID)

	logger.Log().Debugf("Task %s: compacting: start", task.ID)
	filepath, err := compactRequests(cfg.Ammo.Dir, hostname, runner)
	if err != nil {
		return xerrors.Errorf("failed to compact requests: %s", err)
	}
	logger.Log().Debugf("Task %s: compacting: finished", task.ID)

	logger.Log().Debugf("Task %s: sending: start", task.ID)
	if err := sendRequests(&cfg.Scp, filepath, task.ID, runner); err != nil {
		return xerrors.Errorf("failed to send requests: %s", err)
	}
	logger.Log().Debugf("Task %s: sending: finished", task.ID)

	meta := prospectortypes.AmmoPackMetaInfo{
		Born:   born,
		Status: "ok",
	}
	logger.Log().Debugf("Task %s: sending meta: start", task.ID)
	if err := sendMeta(
		&cfg.Scp,
		fmt.Sprintf("%s/%s", cfg.Ammo.Dir, hostname),
		task.ID,
		meta,
		runner); err != nil {
		return xerrors.Errorf("failed to send meta: %s", err)
	}
	logger.Log().Debugf("Task %s: sending meta: finished", task.ID)

	return nil
}

func dumpRequests(cfg *cfgAmmo, duration uint32, hostname string, runner CmdRunner) error {
	cmd := fmt.Sprintf(
		"sudo %s --input-raw :%d --output-file %s/%s --exit-after %ds --output-file-append --output-file-flush-interval 10s --output-file-max-size-limit %d",
		cfg.GorPath,
		cfg.ListenPort,
		cfg.Dir,
		hostname,
		duration,
		cfg.FileSizeLimit,
	)

	_, err := runner.Run(cmd)

	return err
}

func correctPermissions(cfg *cfgAmmo, hostname string, runner CmdRunner) error {
	cmd := fmt.Sprintf("sudo chown %s %s/%s", cfg.LocalUser, cfg.Dir, hostname)
	_, err := runner.Run(cmd)
	return err
}

func compactRequests(ammoDir string, hostname string, runner CmdRunner) (string, error) {
	filepath := fmt.Sprintf("%s/%s", ammoDir, hostname)
	cmd := fmt.Sprintf("gzip %s", filepath)
	_, err := runner.Run(cmd)
	return filepath + ".gz", err
}

func sendRequests(cfg *cfgScp, filename string, taskID string, runner CmdRunner) error {
	return runScp(cfg, filename, taskID, runner)
}

func sendMeta(cfg *cfgScp, filename string, taskID string, meta prospectortypes.AmmoPackMetaInfo, runner CmdRunner) error {
	data, err := json.Marshal(meta)
	if err != nil {
		return xerrors.Errorf("failed to serialize json with meta info: %w", err)
	}

	filenameMeta := filename + prospectortypes.MetaFileSuffix
	err = ioutil.WriteFile(filenameMeta, data, os.FileMode(0660))
	if err != nil {
		return xerrors.Errorf("failed to write meta to disk: %s: %w", filenameMeta, err)
	}

	return runScp(cfg, filenameMeta, taskID, runner)
}

func trySendError(cfg *Config, hostname string, rErr error, taskID string, runner CmdRunner) error {
	filename := fmt.Sprintf("%s/%s", cfg.Ammo.Dir, hostname)
	meta := prospectortypes.AmmoPackMetaInfo{
		Error:  rErr.Error(),
		Status: "error",
	}

	return sendMeta(&cfg.Scp, filename, taskID, meta, runner)
}

func writeMetaToDisk(dir string, meta Meta) error {
	filename := fmt.Sprintf("%s/%s", dir, metaFileName)
	data, err := json.Marshal(meta)
	if err != nil {
		return xerrors.Errorf("failed to serialize json: %w", err)
	}
	return ioutil.WriteFile(filename, data, os.FileMode(0666))
}

func runScp(cfg *cfgScp, filename string, taskID string, runner CmdRunner) error {
	cmd := fmt.Sprintf(
		"scp -o StrictHostKeyChecking=no -i %s -l %d %s %s@%s:%s/%s",
		cfg.PrivateKeyPath,
		cfg.BandwidthKbits,
		filename,
		cfg.RemoteUser,
		cfg.Host,
		cfg.RemoteDir,
		taskID,
	)
	_, err := runner.Run(cmd)
	return err
}
