package ammo

import (
	"fmt"
	"os"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config struct {
	Dir    string   `json:"dir"`
	Hosts  []string `json:"hosts"`
	hosts  map[string]interface{}
	TTL    uint64 `json:"ttl"`
	MinDur uint32 `json:"min_duration"`
	MaxDur uint32 `json:"max_duration"`
}

type State struct {
	mutex       sync.RWMutex
	cfg         Config
	cleanerStop chan bool
	task        *createTask
}

func NewAmmo(cfg Config) (*State, error) {
	if err := checkDir(cfg.Dir); err != nil {
		return nil, err
	}

	res := &State{
		cfg:         cfg,
		cleanerStop: make(chan bool),
	}
	res.cfg.hosts = make(map[string]interface{}, len(res.cfg.Hosts))
	for idx := range res.cfg.Hosts {
		res.cfg.hosts[res.cfg.Hosts[idx]] = nil
	}

	go func() {
		heartbeat := time.NewTicker(1 * time.Hour)

		for {
			select {
			case <-res.cleanerStop:
				logger.Log().Info("Ammo cleaning: quitting goroutine")
				return

			case <-heartbeat.C:
				if err := res.cleanup(); err != nil {
					logger.Log().Warnf("Ammo cleaning: error: %s", err)
				}
			}
		}
	}()

	return res, nil
}

func (s *State) Stop() {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	close(s.cleanerStop)
	if s.task != nil && s.task.stop != nil {
		close(s.task.stop)
	}
}

func checkDir(dir string) error {
	st, err := os.Stat(dir)
	if err != nil {
		return xerrors.Errorf("failed to stat dir for ammo: %s: %w", dir, err)
	}
	if !st.IsDir() {
		return xerrors.Errorf("Path for ammo is not dir: %s", dir)
	}

	fileName := dir + "/.dummy"
	file, err := os.OpenFile(fileName, os.O_RDWR|os.O_CREATE, 0666)
	if err != nil {
		return xerrors.Errorf("failed to check permissions: %s: %w", dir, err)
	}
	_ = file.Close()

	return nil
}

func (s *State) cleanup() error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	logger.Log().Debug("Ammo cleaning: begin")

	packs, err := s.listImpl()
	if err != nil {
		return err
	}

	removalHappend := false
	for k, v := range packs {
		if uint64(time.Now().Unix()) < v.Born+s.cfg.TTL {
			continue
		}
		if s.task != nil && s.task.id == k {
			continue
		}

		removalHappend = true
		if err := os.RemoveAll(s.getPackPath(k)); err != nil {
			logger.Log().Infof("Ammo cleaning: failed to remove: %s: %s", k, err)
		} else {
			logger.Log().Infof("Ammo cleaning: removed: %s", k)
		}
	}

	if !removalHappend {
		logger.Log().Debug("Ammo cleaning: nothing to remove")
	}

	logger.Log().Debug("Ammo cleaning: end")
	return nil
}

func (s *State) getPackPath(id string) string {
	return fmt.Sprintf("%s/%s", s.cfg.Dir, id)
}
