package ammo

import (
	"math/rand"
	"os"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/clitypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

type createTask struct {
	id             string
	hosts          []string
	stop           chan bool
	shouldStop     bool
	isChanelClosed bool
	duration       uint32
}

func (s *State) Create(hosts []string, duration uint32) (*clitypes.AmmoCreateResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if duration < s.cfg.MinDur {
		return nil, xerrors.Errorf("duration is too small: %d vs %d (cfg)", duration, s.cfg.MinDur), nil
	}
	if s.cfg.MaxDur < duration {
		return nil, xerrors.Errorf("duration is too large: %d (cfg) vs %d", s.cfg.MaxDur, duration), nil
	}
	if s.task != nil {
		return nil, xerrors.Errorf("ammo creation is already runnuing: %s", s.task.id), nil
	}

	hosts, err := s.getHosts(hosts)
	if err != nil {
		return nil, err, nil
	}

	id, err := s.createID()
	if err != nil {
		return nil, nil, err
	}

	s.task = &createTask{
		id:       id,
		hosts:    hosts,
		stop:     make(chan bool),
		duration: duration,
	}
	go s.waitCreated()

	return &clitypes.AmmoCreateResult{
		ID: id,
	}, nil, nil
}

func (s *State) getHosts(inList []string) ([]string, error) {
	if len(inList) == 0 {
		return nil, xerrors.New("Hosts list cannot be empty")
	}

	for _, h := range inList {
		if _, found := s.cfg.hosts[h]; !found {
			return nil, xerrors.Errorf("host is unknown: %s", h)
		}
	}

	return inList, nil
}

func (s *State) createID() (string, error) {
	for idx := 0; idx < 50; idx++ {
		rnd := generateRandomString(16)
		path := s.getPackPath(rnd)
		_, err := os.Stat(path)
		if os.IsNotExist(err) {
			// makes because of some reason: drwxr-xr-x
			if err := os.Mkdir(path, os.FileMode(0770)); err != nil {
				logger.Log().Warnf("Ammo creating: failed to create dir: %s: %s", path, err)
				continue
			}

			// makes with setgid: drwxrwx---
			if err := os.Chmod(path, os.FileMode(0770)|os.ModeSetgid); err != nil {
				return "", xerrors.Errorf("Failed to chmod: %w", err)
			}

			return rnd, nil
		}
	}

	return "", xerrors.New("Failed to generate id with 50 tries. Something strange is going on")
}

func (s *State) waitCreated() {
	heartbeat := time.NewTicker(1 * time.Second)

	logger.Log().Info("Ammo creating: started")
	for {
		select {
		case <-s.task.stop:
			logger.Log().Info("Ammo creating: quitting goroutine")
			s.mutex.Lock()
			s.task = nil
			s.mutex.Unlock()
			return

		case <-heartbeat.C:
			if err := s.waitPack(); err != nil {
				logger.Log().Warnf("Ammo creating: error: %s", err)
			}
		}
	}
}

func (s *State) waitPack() error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if s.task == nil {
		panic("task is nil")
	}
	if s.task.shouldStop {
		if !s.task.isChanelClosed {
			logger.Log().Infof("Ammo creating: cancel: %s", s.task.id)
			close(s.task.stop)
			s.task.isChanelClosed = true
		}
		return nil
	}

	pck, err := listPack(s.getPackPath(s.task.id))
	if err != nil {
		return err
	}

	for k := range pck.Hosts {
		foundIdx := -1
		for idx := range s.task.hosts {
			if k == s.task.hosts[idx] {
				foundIdx = idx
				logger.Log().Infof("Ammo creating: pack is ready from %s: %s", k, s.task.id)
				break
			}
		}
		if foundIdx != -1 {
			s.task.hosts = append(s.task.hosts[:foundIdx], s.task.hosts[foundIdx+1:]...)
		}
	}

	if len(s.task.hosts) == 0 && !s.task.isChanelClosed {
		close(s.task.stop)
		s.task.isChanelClosed = true
	}

	return nil
}

var letters = []byte("abcdefghijklmnopqrstuvwxyz")

func generateRandomString(n int) string {
	b := make([]byte, n)
	for i := range b {
		b[i] = letters[rand.Int()%len(letters)]
	}
	return string(b)
}
