package ammo

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/clitypes"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/prospectortypes"
)

type ListResult map[string]clitypes.PackInfo

func (s *State) List() (clitypes.ListResult, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.listImpl()
}

func (s *State) listImpl() (clitypes.ListResult, error) {
	files, err := ioutil.ReadDir(s.cfg.Dir)
	if err != nil {
		return nil, xerrors.Errorf("failed to read dir with ammo ( %s ): %w", s.cfg.Dir, err)
	}

	res := make(clitypes.ListResult)
	for _, d := range files {
		if !d.IsDir() {
			continue
		}

		packInfo, err := listPack(s.getPackPath(d.Name()))
		if err != nil {
			return nil, err
		}

		res[d.Name()] = *packInfo
	}

	if s.task != nil {
		if _, ok := res[s.task.id]; ok {
			if res[s.task.id].StatusInt == clitypes.AmmoReady {
				tmp := res[s.task.id]
				tmp.StatusInt = clitypes.AmmoInProgress
				tmp.Status = clitypes.AmmoInProgress.String()
				res[s.task.id] = tmp
			}
		}
	}

	return res, nil
}

func listPack(packDir string) (*clitypes.PackInfo, error) {
	files, err := ioutil.ReadDir(packDir)
	if err != nil {
		return nil, xerrors.Errorf("failed to read dir with ammo pack ( %s ): %w", packDir, err)
	}

	if len(files) == 0 {
		return &clitypes.PackInfo{Status: clitypes.AmmoInProgress.String(), StatusInt: clitypes.AmmoInProgress}, nil
	}

	res := &clitypes.PackInfo{
		Hosts:     make(map[string]clitypes.HostPackInfo),
		Status:    clitypes.AmmoReady.String(),
		StatusInt: clitypes.AmmoReady,
	}
	for _, d := range files {
		if d.IsDir() || strings.HasSuffix(d.Name(), prospectortypes.MetaFileSuffix) {
			continue
		}

		hostInfo, hostname, err := getHostPackInfo(packDir, d.Name())
		if err != nil {
			return nil, err
		}

		res.Hosts[hostname] = *hostInfo
		if res.StatusInt < hostInfo.StatusInt {
			res.StatusInt = hostInfo.StatusInt
			res.Status = hostInfo.Status
		}
		if res.Born == 0 || (hostInfo.Born != 0 && hostInfo.Born < res.Born) {
			res.Born = hostInfo.Born
		}
		res.Size += hostInfo.Size
	}

	return res, nil
}

func getHostPackInfo(packDir string, filename string) (*clitypes.HostPackInfo, string, error) {
	if !strings.HasSuffix(filename, ".gz") {
		return nil, "", xerrors.Errorf("allowed only ammo in gz: %s", filename)
	}
	hostname := filename[:len(filename)-3]

	filepath := fmt.Sprintf("%s/%s", packDir, filename)
	metaFile := fmt.Sprintf("%s/%s%s", packDir, hostname, prospectortypes.MetaFileSuffix)

	meta, err := getMetaInfo(metaFile)
	if err != nil {
		return nil, hostname, err
	}

	if meta == nil {
		// ammo file exists. Meta file appears afterwards. So copying is in progress.
		return &clitypes.HostPackInfo{Status: clitypes.AmmoInProgress.String(), StatusInt: clitypes.AmmoInProgress}, hostname, nil
	}

	if !strings.EqualFold(meta.Status, "ok") {
		return &clitypes.HostPackInfo{
			Status:    clitypes.AmmoInvalid.String(),
			StatusInt: clitypes.AmmoInProgress,
			Error:     fmt.Sprintf("Invalid ammo pack: %s. Status=%s. Error: %s", metaFile, meta.Status, meta.Error),
		}, hostname, nil
	}

	st, err := os.Stat(filepath)
	if err != nil {
		return nil, hostname, xerrors.Errorf("Failed to stat ammo file: %s: %w", filepath, err)
	}

	return &clitypes.HostPackInfo{
		Status:    clitypes.AmmoReady.String(),
		StatusInt: clitypes.AmmoReady,
		Born:      meta.Born,
		Size:      uint64(st.Size()),
	}, hostname, nil
}

func getMetaInfo(filepath string) (*prospectortypes.AmmoPackMetaInfo, error) {
	_, err := os.Stat(filepath)
	if os.IsNotExist(err) {
		return nil, nil
	}

	metaBody, err := ioutil.ReadFile(filepath)
	if err != nil {
		return nil, err
	}

	var meta prospectortypes.AmmoPackMetaInfo
	if err := json.Unmarshal(metaBody, &meta); err != nil {
		return nil, xerrors.Errorf("failed to parse meta: %s: %w", filepath, err)
	}

	return &meta, nil
}
