package locker

import (
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/types"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Result map[string]string

type Config struct {
	MaxDuration uint32 `json:"max_duration"`
}

type State struct {
	mutex sync.RWMutex
	cfg   Config
	l     *lock
}

type lock struct {
	stop     chan bool
	login    string
	start    time.Time
	duration time.Duration
}

func NewLocker(cfg Config) (*State, error) {
	res := &State{
		cfg: cfg,
	}
	return res, nil
}

func (s *State) Stop() {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if s.l != nil {
		close(s.l.stop)
	}
}

func (s *State) GetStatus() (types.StatusResult, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	if s.l == nil {
		return nil, nil
	}

	return types.StatusResult{
		types.StatusStr: "Locked",
		"user":          s.l.login,
	}, nil
}

func (s *State) Lock(duration uint32, login string) error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if s.l != nil {
		return xerrors.Errorf("shooting gallery already locked by '%s'", s.l.login)
	}
	s.l = &lock{
		stop:     make(chan bool),
		login:    login,
		start:    time.Now(),
		duration: time.Duration(duration) * time.Second,
	}
	logger.Log().Infof("Locker: lock: %s", login)

	go func() {
		heartbeat := time.NewTicker(time.Duration(duration) * time.Second)
		select {
		case <-s.l.stop:
			logger.Log().Info("Locker: quitting")
		case <-heartbeat.C:
			logger.Log().Info("Locker: timeout")
		}

		s.mutex.Lock()
		s.l = nil
		s.mutex.Unlock()
		logger.Log().Infof("Locker: unlock: %s", login)
	}()

	return nil
}

func (s *State) Unlock(login string) error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if s.l == nil {
		return xerrors.Errorf("shooting gallery is not locked")
	}
	if s.l.login != login {
		return xerrors.Errorf("you (%s) cannot unlock: another user created lock: %s", login, s.l.login)
	}
	close(s.l.stop)

	return nil
}

func (s *State) CheckAllowed(login string) error {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	if s.l == nil {
		return xerrors.Errorf("shooting gallery must be locked")
	}
	if s.l.login != login {
		return xerrors.Errorf("shooting gallery is locked since %s for %s: %s", s.l.start, s.l.duration, s.l.login)
	}

	return nil
}
