package responder

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type ctxKey string

const (
	AuthType = ctxKey("AuthType")
	AuthID   = ctxKey("AuthID")
)

type Resp struct {
	logger *log.Logger
}

func NewResponder(accessFilename string) (*Resp, error) {
	f, err := os.OpenFile(accessFilename, os.O_RDWR|os.O_CREATE|os.O_APPEND, 0666)
	if err != nil {
		return nil, xerrors.Errorf("failed to open access log")
	}

	logger := log.New(f, "", log.LstdFlags|log.Lmicroseconds)
	return &Resp{
		logger: logger,
	}, nil
}

func (t *Resp) Return200(w http.ResponseWriter, r *http.Request, printable interface{}) {
	t.returnObject(w, r, printable, http.StatusOK)
}

func (t *Resp) Return400(w http.ResponseWriter, r *http.Request, err error) {
	type errResponse struct {
		Err string `json:"error"`
	}

	t.returnObject(w, r, errResponse{Err: err.Error()}, http.StatusBadRequest)
}

func (t *Resp) Return401(w http.ResponseWriter, r *http.Request, printable interface{}) {
	t.returnObject(w, r, printable, http.StatusUnauthorized)
}

func (t *Resp) Return500(w http.ResponseWriter, r *http.Request, err error) {
	t.log(r, http.StatusServiceUnavailable, []byte(err.Error()))

	w.WriteHeader(http.StatusServiceUnavailable)
	w.Header().Set("Content-Type", "text/plain")
	_, _ = w.Write([]byte(err.Error()))
}

func (t *Resp) returnObject(w http.ResponseWriter, r *http.Request, printable interface{}, httpCode int) {
	rsp, err := json.Marshal(printable)
	if err != nil {
		t.Return500(w, r, xerrors.Errorf("failed to seriazile json: %w", err))
		return
	}

	t.log(r, httpCode, rsp)

	w.WriteHeader(httpCode)
	w.Header().Set("Content-Type", "application/json")
	w.Header().Set("Content-Length", fmt.Sprintf("%d", len(rsp)))
	_, _ = w.Write(rsp)
}

func (t *Resp) log(r *http.Request, httpCode int, resp []byte) {
	if httpCode == http.StatusOK {
		resp = []byte("-")
	}

	t.logger.Printf(
		"%s %s %s %d %s %s %s",
		r.RemoteAddr,
		GetStringFromCtx(r, AuthType),
		GetStringFromCtx(r, AuthID),
		httpCode,
		r.Method,
		r.URL.Path,
		resp,
	)
}

func GetStringFromCtx(r *http.Request, key ctxKey) string {
	v := r.Context().Value(key)
	if v == nil {
		return "-"
	}

	s := v.(string)
	if s == "" {
		s = "-"
	}
	return s
}
