package shooter

import (
	"encoding/json"
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/ammo"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/locker"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/shooting"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/stateviewer"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/types"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/clitypes"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/prospectortypes"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type State struct {
	mutex       sync.RWMutex
	ammo        *ammo.State
	shooting    *shooting.State
	stateviewer *stateviewer.State
	lock        *locker.State
	cfg         Config
}

func NewState(cfg Config, lock *locker.State) (*State, error) {
	amm, err := ammo.NewAmmo(cfg.Ammo)
	if err != nil {
		return nil, err
	}

	st, err := stateviewer.NewStateviewer()
	if err != nil {
		return nil, err
	}

	res := &State{
		ammo:        amm,
		cfg:         cfg,
		stateviewer: st,
		lock:        lock,
	}
	return res, nil
}

func (s *State) Stop() {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	s.ammo.Stop()
}

func (s *State) GetStatus() (types.StatusResult, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	for _, sg := range []func() (types.StatusResult, error){
		func() (types.StatusResult, error) { return s.getStatus() },
		func() (types.StatusResult, error) { return s.lock.GetStatus() },
	} {
		st, err := sg()
		if err != nil {
			return nil, err
		}
		if st != nil {
			return st, nil
		}
	}

	return types.StatusResult{
		types.StatusStr: "Idle",
	}, nil
}

func (s *State) getStatus() (types.StatusResult, error) {
	getters := []types.StatusGetter{
		s.ammo,
	}
	if s.shooting != nil {
		getters = append(getters, s.shooting)
	}

	for _, sg := range getters {
		st, err := sg.GetStatus()
		if err != nil {
			return nil, err
		}
		if st != nil {
			return st, nil
		}
	}

	return nil, nil
}

func (s *State) ShootingStart(params shooting.Params, cmdCr shooting.CmdCreator) (shooting.StartResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if params.Schema != "http" && params.Schema != "https" {
		return nil, xerrors.Errorf("schema is unknown: %s", params.Schema), nil
	}
	if params.Instances == 0 {
		return nil, xerrors.Errorf("instances cannot be 0"), nil
	}
	if params.Duration == 0 {
		params.Duration = shooting.MaxDuration
	}
	if params.Duration > shooting.MaxDuration {
		return nil, xerrors.Errorf("shooting duration is too longer then 1 hour: %d seconds", params.Duration), nil
	}
	if params.Workers == 0 {
		params.Workers = 128
	}

	list, err := s.ammo.List()
	if err != nil {
		return nil, nil, xerrors.Errorf("failed to list ammo: %w", err)
	}
	pack, ok := list[params.AmmoID]
	if !ok {
		return nil, xerrors.Errorf("missing ammo pack: %s", params.AmmoID), nil
	}
	if pack.StatusInt != clitypes.AmmoReady {
		return nil, xerrors.Errorf("ammo pack is not ready: %s", pack.Status), nil
	}

	if !shooting.IsStopped(s.shooting) {
		return nil, xerrors.Errorf("shooting is already running"), nil
	}

	sh, err := shooting.NewShooting(s.cfg.Shooting, params, s.cfg.Ammo.Dir, pack, cmdCr)
	if err != nil {
		return nil, nil, xerrors.Errorf("failed to start shooting: %w", err)
	}

	s.shooting = sh

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) ShootingStop() (shooting.StopResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if shooting.IsStopped(s.shooting) {
		return nil, xerrors.Errorf("nothing to stop"), nil
	}

	s.shooting.Stop()
	s.shooting = nil

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) AmmoList() (clitypes.ListResult, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.ammo.List()
}

func (s *State) AmmoCreate(hosts []string, duration uint32) (*clitypes.AmmoCreateResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	idle, details, err := s.isIdle()
	if err != nil {
		return nil, nil, err
	}
	if !idle {
		return nil, xerrors.Errorf("shooter is not idle: %s", details), nil
	}

	return s.ammo.Create(hosts, duration)
}

func (s *State) AmmoDelete(id string) (ammo.DeleteResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if !shooting.IsStopped(s.shooting) && s.shooting.GetAmmoID() == id {
		return nil, xerrors.Errorf("ammo %s is being using right now", id), nil
	}

	return s.ammo.Delete(id)
}

func (s *State) ProspectorGetTask(host string) (prospectortypes.Task, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.ammo.GetTask(host)
}

func (s *State) GetStateTop() (*stateviewertypes.Top, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	return s.stateviewer.GetTop()
}

func (s *State) PostStateviewerTop(output string, timestamp int64) (stateviewer.PostResult, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	var st types.StatusResult
	if !shooting.IsStopped(s.shooting) {
		var err error
		st, err = s.shooting.GetStatus()
		if err != nil {
			logger.Log().Warnf("PostStateviewerTop: failed to get status for shooting: %s", err)
		}
	}
	s.stateviewer.PostTop(output, timestamp, st)

	return map[string]string{
		"status": "ok",
	}, nil
}

func (s *State) GetStatePerf() (*stateviewertypes.Perf, error, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	p, err := s.stateviewer.GetPerf()
	return p, err, nil
}

func (s *State) PostStatePerf(output string, timestamp int64) (stateviewer.PostResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	var st types.StatusResult
	if !shooting.IsStopped(s.shooting) {
		var err error
		st, err = s.shooting.GetStatus()
		if err != nil {
			logger.Log().Warnf("PostStateviewerTop: failed to get status for shooting: %s", err)
		}
	}

	s.stateviewer.AddPerf(st, output, timestamp)

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) CancelStatePerf() (stateviewer.PostResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if err := s.stateviewer.CancelPerfCmd(); err != nil {
		return nil, err, nil
	}

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) GetVersion() (stateviewertypes.Versions, error, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	res, err := s.stateviewer.GetVersions()
	if err != nil {
		return nil, nil, err
	}

	return res, nil, nil
}

func (s *State) PostInstallVersionCmd(pack string, version string) (stateviewer.PostResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	rErr, err := s.stateviewer.AddInstallVersionCmd(pack, version)
	if err != nil {
		return nil, nil, err
	}
	if rErr != nil {
		return nil, rErr, nil
	}

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) GetStatePerfCmd() (stateviewertypes.PerfCmd, error, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.stateviewer.GetPerfCmd(), nil, nil
}

func (s *State) PostStatePerfCmd(frequency uint32, sleep uint32) (stateviewer.PostResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if frequency > s.cfg.Shooting.PerfRecorf.MaxFrequency {
		return nil, xerrors.Errorf("frequency is too high: %d. Max allowed: %d", frequency, s.cfg.Shooting.PerfRecorf.MaxFrequency), nil
	}
	if sleep > s.cfg.Shooting.PerfRecorf.MaxSleep {
		return nil, xerrors.Errorf("sleep is too long: %d. Max allowed: %d", sleep, s.cfg.Shooting.PerfRecorf.MaxSleep), nil
	}

	if frequency == 0 {
		frequency = 999
	}
	if sleep == 0 {
		sleep = 30
	}

	if err := s.stateviewer.AddPerfCmd(frequency, sleep); err != nil {
		return nil, err, nil
	}

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) GetInstallVersionCmd() (stateviewertypes.Versions, error, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.stateviewer.GetInstallVersionCmd(), nil, nil
}

func (s *State) PostVersion(data []byte) (stateviewer.PostResult, error, error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	var vers stateviewertypes.Versions
	if err := json.Unmarshal(data, &vers); err != nil {
		return nil, xerrors.Errorf("failed to parse versions: %s", err), nil
	}

	s.stateviewer.SetVersions(vers)

	return map[string]string{
		"status": "ok",
	}, nil, nil
}

func (s *State) isIdle() (bool, string, error) {
	st, err := s.getStatus()
	if err != nil {
		return false, "", err
	}

	if st == nil {
		return true, "", nil
	}

	status := st["status"].(string)
	return false, status, nil
}
