package shooting

import (
	"fmt"
	"sync"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/types"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/clitypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config struct {
	GorPath    string           `json:"gor_path"`
	Target     string           `json:"target"`
	PerfRecorf PerfRecordConfig `json:"perf_record"`
}

type PerfRecordConfig struct {
	MaxFrequency uint32 `json:"max_frequency"`
	MaxSleep     uint32 `json:"max_sleep"`
}

const MaxDuration = 3600

type State struct {
	mutex   sync.RWMutex
	wg      sync.WaitGroup
	stopped bool

	gors []Cmd

	cfg       Config
	params    Params
	packInfo  clitypes.PackInfo
	startTime int64
}

type StartResult map[string]string
type StopResult map[string]string

type Params struct {
	AmmoID          string `json:"ammo_id"`
	Schema          string `json:"schema"`
	Rate            uint32 `json:"rate"`
	Instances       uint32 `json:"instances"`
	Duration        uint32 `json:"duration"`
	Workers         uint32 `json:"workers"`
	ConnectionClose bool   `json:"connection_close"`
}

func IsStopped(s *State) bool {
	if s == nil {
		return true
	}

	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.stopped
}

func NewShooting(cfg Config, params Params, ammoDir string, info clitypes.PackInfo, cmdCr CmdCreator) (*State, error) {
	res := &State{
		gors:      make([]Cmd, 0, params.Instances),
		cfg:       cfg,
		params:    params,
		packInfo:  info,
		startTime: time.Now().Unix(),
	}

	for host := range info.Hosts {
		for idx := uint32(0); idx < params.Instances; idx++ {
			if err := res.runGoreplay(ammoDir, host, idx, cmdCr); err != nil {
				return nil, xerrors.Errorf("failed to start goreplay: %w", err)
			}
		}
	}
	return res, nil
}

func (s *State) GetStatus() (types.StatusResult, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	if s.stopped {
		return nil, nil
	}

	return types.StatusResult{
		types.StatusStr: "Shooting",
		"ammo_info":     s.packInfo,
		"params":        s.params,
		"start_time":    s.startTime,
	}, nil
}

func (s *State) Stop() {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	for idx := range s.gors {
		logger.Log().Infof("Stooting #%s: %d: stopping", s.params.AmmoID, idx)
		if err := s.gors[idx].Signal(syscall.SIGTERM); err != nil {
			logger.Log().Errorf("Stooting #%s: %d: failed to send SIGTERM: %s", s.params.AmmoID, idx, err)
		}
	}

	s.wg.Wait()
	s.stopped = true
}

func (s *State) GetAmmoID() string {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.params.AmmoID
}

func (s *State) runGoreplay(ammoDir string, host string, idx uint32, cmdCr CmdCreator) error {
	cmdStr := fmt.Sprintf(
		`%s --input-file-loop --input-file %s/%s/%s.gz|%d%% --output-http-workers=%d --output-http %s://%s --exit-after %ds`,
		s.cfg.GorPath,
		ammoDir,
		s.params.AmmoID,
		host,
		s.params.Rate,
		s.params.Workers,
		s.params.Schema,
		s.cfg.Target,
		s.params.Duration,
	)
	if s.params.ConnectionClose {
		cmdStr = cmdStr + ` --http-set-header "Connection: close"`
	}

	logger.Log().Infof("Stooting #%s: %d: starting goreplay: %s", s.params.AmmoID, idx, cmdStr)

	cmd := cmdCr.Create(cmdStr)
	s.gors = append(s.gors, cmd)

	s.wg.Add(1)
	go func() {
		out, err := cmd.Run()
		if err != nil {
			logger.Log().Errorf("Stooting #%s: %s(%d): error on Wait(): %s: %s", s.params.AmmoID, host, idx, err, string(out))
		}
		logger.Log().Infof("Stooting #%s: %s(%d): stopped", s.params.AmmoID, host, idx)

		s.wg.Done()
		s.cleanup(cmd.GetPid())
	}()

	return nil
}

func (s *State) cleanup(pid int) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	foundIdx := -1
	for idx := range s.gors {
		if pid == s.gors[idx].GetPid() {
			foundIdx = idx
			break
		}
	}
	if foundIdx != -1 {
		s.gors = append(s.gors[:foundIdx], s.gors[foundIdx+1:]...)
	}

	if len(s.gors) == 0 {
		s.stopped = true
	}
}

func (s *State) isStopped() bool {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	return s.stopped
}
