package srv

import (
	"encoding/json"
	"io/ioutil"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/shooter"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type config struct {
	Log           logger.Config     `json:"logger"`
	HTTPConfig    httpCommonConfig  `json:"http_common"`
	HTTPUnistat   httpUnistatConfig `json:"http_unistat"`
	ShooterConfig shooter.Config    `json:"shooter"`
}

func Run(cfgFile string) error {
	cfg, err := parseConfig(cfgFile)
	if err != nil {
		return err
	}

	if err := logger.Init(cfg.Log); err != nil {
		return err
	}

	logger.Log().Info("Starting shooter")
	sh, err := shooter.NewShooter(cfg.ShooterConfig)
	if err != nil {
		logger.Log().Infof("Failed to start shooter: %s", err)
		return err
	}

	stopChannel := make(chan bool)

	logger.Log().Info("Starting http")
	httpCommon, err := startCommonHTTP(cfg.HTTPConfig, stopChannel, sh)
	if err != nil {
		logger.Log().Infof("Failed to start http common: %s", err)
		return err
	}
	httpUnistat, err := startUnistatHTTP(cfg.HTTPUnistat, stopChannel)
	if err != nil {
		logger.Log().Infof("Failed to start http unistat: %s", err)
		return err
	}

	logger.Log().Info("Started")

	signalChannel := make(chan os.Signal, 10)
	signal.Notify(signalChannel, syscall.SIGINT, syscall.SIGTERM)

	select {
	case sig := <-signalChannel:
		switch sig {
		case syscall.SIGTERM:
			logger.Log().Info("Stopping: SIGTERM")
		case syscall.SIGINT:
			logger.Log().Info("Stopping: SIGINT")
		}

	case <-stopChannel:
		logger.Log().Info("Stopping: internal reason")
	}

	httpUnistat.stop()
	httpCommon.stop()
	sh.Stop()

	return nil
}

func parseConfig(cfgFile string) (*config, error) {
	data, err := ioutil.ReadFile(cfgFile)
	if err != nil {
		return nil, err
	}

	var cfg config
	if err := json.Unmarshal(data, &cfg); err != nil {
		return nil, xerrors.Errorf("failed to parse config: %w", err)
	}

	return &cfg, nil
}
