package stateviewer

import (
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/types"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
)

type PostResult map[string]string

type State struct {
	topMutex sync.RWMutex
	top      *stateviewertypes.Top

	perfMutex sync.RWMutex
	perf      *stateviewertypes.Perf
	perfCmd   stateviewertypes.PerfCmd

	versionMutex sync.RWMutex
	versions     stateviewertypes.Versions
	versionsCmd  stateviewertypes.Versions
}

func NewStateviewer() (*State, error) {
	return &State{
		versions:    make(stateviewertypes.Versions),
		versionsCmd: make(stateviewertypes.Versions),
	}, nil
}

func (s *State) PostTop(output string, timestamp int64, status types.StatusResult) {
	s.topMutex.Lock()
	defer s.topMutex.Unlock()

	s.top = &stateviewertypes.Top{
		Output:    output,
		Timestamp: timestamp,
		Shooting:  status,
	}
}

func (s *State) GetTop() (*stateviewertypes.Top, error) {
	s.topMutex.RLock()
	defer s.topMutex.RUnlock()

	if s.top == nil {
		return nil, xerrors.Errorf("there is no top in memory")
	}

	return s.top, nil
}

func (s *State) AddPerfCmd(frequency uint32, sleep uint32) error {
	s.perfMutex.Lock()
	defer s.perfMutex.Unlock()

	if s.perfCmd.NeedCreate {
		return xerrors.Errorf("perf is already being coollecting now")
	}

	s.perf = nil
	s.perfCmd = stateviewertypes.PerfCmd{
		NeedCreate: true,
		Frequency:  frequency,
		Sleep:      sleep,
	}
	return nil
}

func (s *State) GetPerfCmd() stateviewertypes.PerfCmd {
	s.perfMutex.RLock()
	defer s.perfMutex.RUnlock()

	return s.perfCmd
}

func (s *State) CancelPerfCmd() error {
	s.perfMutex.Lock()
	defer s.perfMutex.Unlock()

	if !s.perfCmd.NeedCreate {
		return xerrors.Errorf("there is no perf task to cancel")
	}
	s.perfCmd = stateviewertypes.PerfCmd{}

	return nil
}

func (s *State) AddPerf(shooting types.StatusResult, output string, timestamp int64) {
	s.perfMutex.Lock()
	defer s.perfMutex.Unlock()

	s.perf = &stateviewertypes.Perf{
		Output:    output,
		Timestamp: timestamp,
		Shooting:  shooting,
	}
	s.perfCmd = stateviewertypes.PerfCmd{}
}

func (s *State) GetPerf() (*stateviewertypes.Perf, error) {
	s.perfMutex.RLock()
	defer s.perfMutex.RUnlock()

	if s.perfCmd.NeedCreate {
		return nil, xerrors.Errorf("perf data is still being collecting")
	}
	if s.perf == nil {
		return nil, xerrors.Errorf("there is no perf data in memory")
	}

	return s.perf, nil
}

func (s *State) GetVersions() (stateviewertypes.Versions, error) {
	s.versionMutex.RLock()
	defer s.versionMutex.RUnlock()

	if len(s.versions) == 0 {
		return nil, xerrors.Errorf("there is no versions info in memory")
	}

	return s.versions, nil
}

func (s *State) SetVersions(vers stateviewertypes.Versions) {
	s.versionMutex.Lock()
	defer s.versionMutex.Unlock()

	s.versions = vers

	// cleanup cmd - maybe it is done
	for k, v := range vers {
		cmdVer, ok := s.versionsCmd[k]
		if ok && cmdVer == v {
			delete(s.versionsCmd, k)
		}
	}
}

func (s *State) AddInstallVersionCmd(pack string, version string) (error, error) {
	s.versionMutex.Lock()
	defer s.versionMutex.Unlock()

	if len(s.versions) == 0 {
		return nil, xerrors.Errorf("there is no versions info in memory")
	}
	if _, ok := s.versions[pack]; !ok {
		return xerrors.Errorf("unknown package: %s", pack), nil
	}

	s.versionsCmd[pack] = version
	return nil, nil
}

func (s *State) GetInstallVersionCmd() stateviewertypes.Versions {
	s.versionMutex.RLock()
	defer s.versionMutex.RUnlock()

	return s.versionsCmd
}
