package tvmcore

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/internal/responder"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type State struct {
	tp2          *tvmauth.Client
	allowedTvmID map[tvm.ClientID]interface{}
	responder    *responder.Resp
}

type Config struct {
	ClientPath   string         `json:"client"`
	CacheDir     string         `json:"cache_dir"`
	StaffTvmID   tvm.ClientID   `json:"staff_tvm_id"`
	AllowedTvmID []tvm.ClientID `json:"allowed_tvm_id"`
	TvmAPIHost   string         `json:"tvm_api_host"`
	TvmAPIPort   uint16         `json:"tvm_api_port"`
}

type tvmClient struct {
	ID     tvm.ClientID `json:"id"`
	Secret string       `json:"secret"`
}

const (
	TvmAliasStaff = "STAFF"
)

func NewTvmCore(cfg Config, resp *responder.Resp) (*State, error) {
	tp2, err := initTvm(cfg)
	if err != nil {
		return nil, xerrors.Errorf("failed to init tvm client: %w", err)
	}

	res := &State{
		tp2:          tp2,
		allowedTvmID: make(map[tvm.ClientID]interface{}, len(cfg.AllowedTvmID)),
		responder:    resp,
	}

	for idx := range cfg.AllowedTvmID {
		res.allowedTvmID[cfg.AllowedTvmID[idx]] = nil
	}

	return res, nil
}

func (s *State) Stop() {
	s.tp2.Destroy()
}

func (s *State) GetTicket(alias string) (string, error) {
	return s.tp2.GetServiceTicketForAlias(context.Background(), alias)
}

func (s *State) Middleware(next http.HandlerFunc) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		r = r.WithContext(context.WithValue(r.Context(), responder.AuthType, "tvm"))

		st := r.Header.Get("X-Ya-Service-Ticket")
		if st == "" {
			s.responder.Return401(w, r, map[string]string{
				"error": "missing service ticket",
			})
			return
		}

		ticket, err := s.tp2.CheckServiceTicket(r.Context(), st)
		if err != nil {
			res := map[string]string{
				"error":  "service ticket is invalid",
				"desc":   err.Error(),
				"status": err.(*tvm.TicketError).Status.String(),
			}
			if ticket != nil {
				res["debug_info"] = ticket.DbgInfo
			}
			s.responder.Return401(w, r, res)
			return
		}

		r = r.WithContext(context.WithValue(r.Context(), responder.AuthID, fmt.Sprintf("%d", ticket.SrcID)))

		if _, ok := s.allowedTvmID[ticket.SrcID]; !ok {
			s.responder.Return401(w, r, map[string]string{
				"error": fmt.Sprintf("tvm_id=%d is not allowed", ticket.SrcID),
			})
			return
		}

		next.ServeHTTP(w, r)
	}
}

func initTvm(cfg Config) (*tvmauth.Client, error) {
	data, err := ioutil.ReadFile(cfg.ClientPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to read file with tvm client: %s: %s", cfg.ClientPath, err)
	}

	var clientCfg tvmClient
	if err := json.Unmarshal(data, &clientCfg); err != nil {
		return nil, xerrors.Errorf("failed to parse file with tvm client: %s: %s", cfg.ClientPath, err)
	}

	settings := tvmauth.TvmAPISettings{
		SelfID:                      clientCfg.ID,
		EnableServiceTicketChecking: true,
		DiskCacheDir:                cfg.CacheDir,
		TVMHost:                     cfg.TvmAPIHost,
		TVMPort:                     int(cfg.TvmAPIPort),
	}
	settings.ServiceTicketOptions = tvmauth.NewAliasesOptions(
		clientCfg.Secret,
		map[string]tvm.ClientID{
			TvmAliasStaff: cfg.StaffTvmID,
		},
	)

	return tvmauth.NewAPIClient(settings, logger.Log().With(log.String("TvmClient", "")))
}
