package perf

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/tvmclient"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config struct {
	Dir     string `json:"tmp_dir"`
	PidFile string `json:"pid_file"`
}

type State struct {
	stop chan bool
}

func NewPerfer(cfg Config, tvm tvmclient.Tvm, shooterURL string) (*State, error) {
	hc := &misc.HTTPClientImpl{
		Client: resty.New().
			SetHostURL(shooterURL).
			SetTimeout(5 * time.Second).
			SetRedirectPolicy(resty.NoRedirectPolicy()),
	}
	cmd := &misc.CmdRunnerImpl{}

	res := &State{
		stop: make(chan bool),
	}

	go func() {
		heartbeat := time.NewTicker(5 * time.Second)

		for {
			select {
			case <-res.stop:
				logger.Log().Info("Perf: quitting")
				return

			case <-heartbeat.C:
				if err := run(cfg, hc, cmd, tvm, time.Now().Unix()); err != nil {
					logger.Log().Warnf("Perf: error: %s", err)
				}
			}
		}
	}()

	return res, nil
}

func (s *State) Stop() {
	close(s.stop)
}

func run(cfg Config, shooter misc.HTTPClient, cmd misc.CmdRunner, tvm tvmclient.Tvm, now int64) error {
	ticket, err := tvm.GetServiceTicket(tvmclient.TvmAliasShooter)
	if err != nil {
		return xerrors.Errorf("failed to get service ticket: %s", err)
	}
	authHeaders := map[string]string{"X-Ya-Service-Ticket": ticket}

	task, err := getTask(shooter, authHeaders)
	if err != nil {
		return err
	}

	if !task.NeedCreate {
		return nil
	}

	return performTask(cfg, shooter, cmd, now, task, authHeaders)
}

func getTask(shooter misc.HTTPClient, authHeaders map[string]string) (*stateviewertypes.PerfCmd, error) {
	code, taskBytes, err := shooter.Get("/stateviewer/perf", authHeaders)
	if err != nil || code != 200 {
		return nil, xerrors.Errorf("failed to get task: %d: %s: %s", code, string(taskBytes), err)
	}

	var task stateviewertypes.PerfCmd
	if err := json.Unmarshal(taskBytes, &task); err != nil {
		return nil, xerrors.Errorf("failed to parse task: %d: %s: %s", code, string(taskBytes), err)
	}

	return &task, nil
}

func performTask(cfg Config, shooter misc.HTTPClient, cmd misc.CmdRunner, now int64, task *stateviewertypes.PerfCmd, authHeaders map[string]string) error {
	cmdRecord := fmt.Sprintf(
		"sudo perf record -o %s/some.perf.data -p $(cat %s) -F %d -a -g -- sleep %d",
		cfg.Dir,
		cfg.PidFile,
		task.Frequency,
		task.Sleep,
	)
	perfOut, err := cmd.Run(cmdRecord)
	if err != nil {
		return xerrors.Errorf("failed to run perf recorf: %s: %s", err, perfOut)
	}

	cmdScript := fmt.Sprintf("sudo perf script -i %s/some.perf.data", cfg.Dir)
	perfOut, err = cmd.Run(cmdScript)
	if err != nil {
		return xerrors.Errorf("failed to run perf script: %s: %s", err, perfOut)
	}

	buf, err := misc.Compress(perfOut)
	if err != nil {
		return xerrors.Errorf("failed to compress perf: %w", err)
	}

	path := fmt.Sprintf("/stateviewer/perf?timestamp=%d", now)
	code, resp, err := shooter.Post(path, misc.EncodeBody(buf), authHeaders)
	if err != nil || code != 200 {
		return xerrors.Errorf("failed to push perf: %d: %s: %s", code, string(resp), err)
	}

	return nil
}
