package srv

import (
	"encoding/json"
	"io/ioutil"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/stateviewer"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type config struct {
	Log               logger.Config      `json:"logger"`
	StateviewerConfig stateviewer.Config `json:"stateviewer"`
}

func Run(cfgFile string) error {
	cfg, err := parseConfig(cfgFile)
	if err != nil {
		return err
	}

	if err := logger.Init(cfg.Log); err != nil {
		return err
	}

	logger.Log().Info("Starting stateviewer")
	sh, err := stateviewer.NewStateviewer(cfg.StateviewerConfig)
	if err != nil {
		logger.Log().Infof("Failed to start stateviewer: %s", err)
		return err
	}

	logger.Log().Info("Started")

	signalChannel := make(chan os.Signal, 2)
	signal.Notify(signalChannel, syscall.SIGINT, syscall.SIGTERM)

	sig := <-signalChannel
	switch sig {
	case syscall.SIGTERM:
		logger.Log().Info("Stopping: SIGTERM")
	case syscall.SIGINT:
		logger.Log().Info("Stopping: SIGINT")
	}

	sh.Stop()

	return nil
}

func parseConfig(cfgFile string) (*config, error) {
	data, err := ioutil.ReadFile(cfgFile)
	if err != nil {
		return nil, err
	}

	var cfg config
	if err := json.Unmarshal(data, &cfg); err != nil {
		return nil, xerrors.Errorf("failed to parse config: %w", err)
	}

	return &cfg, nil
}
