package top

import (
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/tvmclient"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config struct {
	Period int64 `json:"period"`
}

type State struct {
	stop chan bool
}

func NewTopper(cfg Config, tvm tvmclient.Tvm, shooterURL string) (*State, error) {
	if cfg.Period == 0 {
		return nil, xerrors.Errorf("Period for pushing top cannot be 0")
	}

	hc := &misc.HTTPClientImpl{
		Client: resty.New().
			SetHostURL(shooterURL).
			SetTimeout(5 * time.Second).
			SetRedirectPolicy(resty.NoRedirectPolicy()),
	}
	cmd := &misc.CmdRunnerImpl{}
	if err := pushTop(hc, cmd, tvm, time.Now().Unix()); err != nil {
		return nil, xerrors.Errorf("Failed to push top: %w", err)
	}

	res := &State{
		stop: make(chan bool),
	}

	go func() {
		heartbeat := time.NewTicker(time.Duration(cfg.Period) * time.Second)

		for {
			select {
			case <-res.stop:
				logger.Log().Info("Top: quitting")
				return

			case <-heartbeat.C:
				if err := pushTop(hc, cmd, tvm, time.Now().Unix()); err != nil {
					logger.Log().Warnf("Top: error: %s", err)
				}
			}
		}
	}()

	return res, nil
}

func (s *State) Stop() {
	close(s.stop)
}

func pushTop(shooter misc.HTTPClient, cmd misc.CmdRunner, tvm tvmclient.Tvm, now int64) error {
	data, err := cmd.Run("top -b -n2 -w 512 -c")
	if err != nil {
		return xerrors.Errorf("failed to run top: %s: %s", err, string(data))
	}

	ticket, err := tvm.GetServiceTicket(tvmclient.TvmAliasShooter)
	if err != nil {
		return xerrors.Errorf("failed to get service ticket: %s", err)
	}

	buf, err := misc.Compress(data)
	if err != nil {
		return xerrors.Errorf("failed to compress perf: %w", err)
	}

	path := fmt.Sprintf("/stateviewer/top?timestamp=%d", now)
	code, resp, err := shooter.Post(path, misc.EncodeBody(buf), map[string]string{
		"X-Ya-Service-Ticket": ticket,
	})
	if err != nil || code != 200 {
		return xerrors.Errorf("failed to push top: %d: %s: %s", code, string(resp), err)
	}

	return nil
}
