package tvmclient

import (
	"context"
	"encoding/json"
	"io/ioutil"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config struct {
	ClientPath   string       `json:"client"`
	CacheDir     string       `json:"cache_dir"`
	ShooterTvmID tvm.ClientID `json:"shooter_tvm_id"`
}

type tvmClient struct {
	ID     tvm.ClientID `json:"id"`
	Secret string       `json:"secret"`
}

type Tvm interface {
	GetServiceTicket(alias string) (string, error)
}

type TvmImpl struct {
	tp2 *tvmauth.Client
}

const (
	TvmAliasShooter = "SHOOTER"
)

func NewTvm(cfg Config) (*TvmImpl, error) {
	tp2, err := initTvm(cfg)
	if err != nil {
		return nil, err
	}

	return &TvmImpl{
		tp2: tp2,
	}, nil
}

func (t *TvmImpl) Stop() {
	t.tp2.Destroy()
}

func (t *TvmImpl) GetServiceTicket(alias string) (string, error) {
	return t.tp2.GetServiceTicketForAlias(context.Background(), alias)
}

func initTvm(cfg Config) (*tvmauth.Client, error) {
	data, err := ioutil.ReadFile(cfg.ClientPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to read file with tvm client: %s: %s", cfg.ClientPath, err)
	}

	var clientCfg tvmClient
	if err := json.Unmarshal(data, &clientCfg); err != nil {
		return nil, xerrors.Errorf("failed to parse file with tvm client: %s: %s", cfg.ClientPath, err)
	}

	settings := tvmauth.TvmAPISettings{
		SelfID: clientCfg.ID,
		ServiceTicketOptions: tvmauth.NewAliasesOptions(
			clientCfg.Secret,
			map[string]tvm.ClientID{
				TvmAliasShooter: cfg.ShooterTvmID,
			},
		),
		DiskCacheDir: cfg.CacheDir,
	}

	return tvmauth.NewAPIClient(settings, logger.Log().With(log.String("TvmClient", "")))
}
