package versions

import (
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/shooter/pkg/stateviewertypes"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/misc"
	"a.yandex-team.ru/passport/infra/daemons/shooting_gallery/stateviewer/internal/tvmclient"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Config []string

type State struct {
	stop chan bool
}

func NewVersioner(cfg Config, tvm tvmclient.Tvm, shooterURL string) (*State, error) {
	hc := &misc.HTTPClientImpl{
		Client: resty.New().
			SetHostURL(shooterURL).
			SetTimeout(5 * time.Second).
			SetRedirectPolicy(resty.NoRedirectPolicy()),
	}
	cmd := &misc.CmdRunnerImpl{}

	res := &State{
		stop: make(chan bool),
	}

	go func() {
		heartbeat := time.NewTicker(10 * time.Second)

		for {
			select {
			case <-res.stop:
				logger.Log().Info("Versioner: quitting")
				return

			case <-heartbeat.C:
				if err := run(cfg, hc, cmd, tvm); err != nil {
					logger.Log().Warnf("Versioner: error: %s", err)
				}
			}
		}
	}()

	return res, nil
}

func (s *State) Stop() {
	close(s.stop)
}

func run(cfg Config, shooter misc.HTTPClient, cmd misc.CmdRunner, tvm tvmclient.Tvm) error {
	ticket, err := tvm.GetServiceTicket(tvmclient.TvmAliasShooter)
	if err != nil {
		return xerrors.Errorf("failed to get service ticket: %s", err)
	}
	authHeaders := map[string]string{"X-Ya-Service-Ticket": ticket}

	current, err := getCurrentVersions(cfg, cmd)
	if err != nil {
		return xerrors.Errorf("failed to get current versions: %s", err)
	}

	task, err := getInstallTask(shooter, authHeaders)
	if err != nil {
		logger.Log().Warnf("Versioner: failed to get installing task: %s", err)
	}

	if len(task) > 0 {
		if err := install(task, cmd); err != nil {
			return xerrors.Errorf("failed to install: %s", err)
		}

		current, err = getCurrentVersions(cfg, cmd)
		if err != nil {
			return xerrors.Errorf("failed to get current versions after install: %s", err)
		}
	}

	if err := pushVersions(current, shooter, authHeaders); err != nil {
		return xerrors.Errorf("failed to push current versions: %s", err)
	}

	return nil
}

func getCurrentVersions(cfg Config, cmd misc.CmdRunner) (stateviewertypes.Versions, error) {
	cmdStr := fmt.Sprintf(
		"dpkg -l | grep -E '(%s)'",
		strings.Join(cfg, "|"),
	)

	output, err := cmd.Run(cmdStr)
	if err != nil {
		return nil, xerrors.Errorf("failed to run dpkg: %s", err)
	}

	return parseDpkgOutput(cfg, output)
}

func getInstallTask(shooter misc.HTTPClient, authHeaders map[string]string) (stateviewertypes.Versions, error) {
	code, taskBytes, err := shooter.Get("/stateviewer/version", authHeaders)
	if err != nil || code != 200 {
		return nil, xerrors.Errorf("failed to get task: %d: %s: %s", code, string(taskBytes), err)
	}

	var task stateviewertypes.Versions
	if err := json.Unmarshal(taskBytes, &task); err != nil {
		return nil, xerrors.Errorf("failed to parse task: %d: %s: %s", code, string(taskBytes), err)
	}

	return task, nil
}

func install(vers stateviewertypes.Versions, cmd misc.CmdRunner) error {
	output, err := cmd.Run("sudo apt-get update -qq")
	if err != nil {
		logger.Log().Warnf("Versioner: failed to update packages: %s", output)
	}

	for k, v := range vers {
		cmdInstall := fmt.Sprintf("sudo apt-get install -y --allow-downgrades %s=%s", k, v)
		output, err = cmd.Run(cmdInstall)
		if err != nil {
			return xerrors.Errorf("failed to install package: %s: %s: %s", cmdInstall, err, output)
		}
	}

	output, err = cmd.Run("sudo /etc/init.d/sezam-service restart")
	if err != nil {
		return xerrors.Errorf("failed to restart service: %s: %s", err, output)
	}

	return nil
}

func pushVersions(vers stateviewertypes.Versions, shooter misc.HTTPClient, authHeaders map[string]string) error {
	body, err := json.Marshal(vers)
	if err != nil {
		return xerrors.Errorf("failed to serialize task: %s: %s", string(body), err)
	}

	code, taskBytes, err := shooter.Post("/stateviewer/version", body, authHeaders)
	if err != nil || code != 200 {
		return xerrors.Errorf("failed to push versions: %d: %s: %s", code, string(taskBytes), err)
	}

	return nil
}

func parseDpkgOutput(cfg Config, output []byte) (stateviewertypes.Versions, error) {
	cfgVers := map[string]interface{}{}
	for idx := range cfg {
		cfgVers[cfg[idx]] = nil
	}

	res := stateviewertypes.Versions{}
	for _, line := range strings.Split(string(output), "\n") {
		pkg, ver, ok := parseDpkgLine(cfgVers, line)
		if ok {
			res[pkg] = ver
		}
	}

	return res, nil
}

func parseDpkgLine(cfg map[string]interface{}, line string) (string, string, bool) {
	if !strings.HasPrefix(line, "ii") { // is package installed
		return "", "", false
	}

	tokens := strings.Split(line, " ")
	tokens = tokens[1:] // skip 'ii'

	pkg := ""
	for _, token := range tokens {
		if token == "" {
			continue
		}

		if pkg == "" {
			if _, ok := cfg[token]; !ok {
				return "", "", false
			}
			pkg = token
			continue
		}

		// token is version
		return pkg, token, true
	}

	return "", "", false
}
