package unittest

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/tirole/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tirole/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/keys"
)

type Config struct {
	RolesDir string `json:"roles_dir"`
}

type FakeProvider struct {
	mappings      map[string]model.Mapping
	preparedRoles map[string]preparedRoles
}

func NewProvider(cfg Config, keymap *keys.KeyMap) (*FakeProvider, error) {
	mappings, err := readMapping(cfg.RolesDir)
	if err != nil {
		return nil, xerrors.Errorf("failed to build slug->tvmid mappings from dir: %v", err)
	}

	requiredSlugs := make([]string, 0, len(mappings))
	for k := range mappings {
		requiredSlugs = append(requiredSlugs, k)
	}
	prepared, err := buildPreparedRoles(cfg.RolesDir, requiredSlugs, keymap)
	if err != nil {
		return nil, xerrors.Errorf("failed to build prepared roles from dir: %v", err)
	}

	return &FakeProvider{
		mappings:      mappings,
		preparedRoles: prepared,
	}, nil
}

func (p *FakeProvider) Ping(context.Context) error {
	return nil
}

func (p *FakeProvider) GetActualRevision(ctx context.Context, slug string) (string, error) {
	roles, found := p.preparedRoles[slug]
	if !found {
		return "", &errs.NoRolesError{
			Message: fmt.Sprintf("There are no roles for slug=%s", slug),
		}
	}

	return roles.meta.RevisionExt, nil
}

func (p *FakeProvider) GetActualRoles(ctx context.Context, slug string) (model.ActualRoles, error) {
	roles, found := p.preparedRoles[slug]
	if !found {
		return model.ActualRoles{}, &errs.NoRolesError{
			Message: fmt.Sprintf("There are no roles for slug=%s", slug),
		}
	}

	return model.ActualRoles{
		Meta: roles.meta,
		Blob: roles.compressed,
	}, nil
}

func (p *FakeProvider) GetMapping(ctx context.Context, slug string) (model.Mapping, error) {
	res, found := p.mappings[slug]
	if !found {
		return make(model.Mapping), nil
	}

	return res, nil
}
