package unittest

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/hex"
	"fmt"
	"io/ioutil"

	"github.com/andybalholm/brotli"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tirole/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/keys"
)

type preparedRoles struct {
	structuredRoles *tvm.Roles
	compressed      []byte
	meta            model.Meta
}

func buildPreparedRoles(dir string, requiredSlugs []string, keymap *keys.KeyMap) (map[string]preparedRoles, error) {
	roles, err := readRoles(dir, requiredSlugs)
	if err != nil {
		return nil, xerrors.Errorf("failed to build slug->tvmid roles from dir: %v", err)
	}

	res := make(map[string]preparedRoles)
	for slug, rr := range roles {
		prepared, err := compressAndSignRoles(rr, keymap)
		if err != nil {
			return nil, xerrors.Errorf("failed to prepare blob for slug %s: %v", slug, err)
		}
		res[slug] = prepared
	}

	return res, nil
}

func readRoles(dir string, requiredSlugs []string) (map[string]*tvm.Roles, error) {
	res := make(map[string]*tvm.Roles)

	for _, slug := range requiredSlugs {
		filename := fmt.Sprintf("%s%s.json", dir, slug)

		body, err := ioutil.ReadFile(filename)
		if err != nil {
			return nil, xerrors.Errorf("failed to read file body '%s': %v", filename, err)
		}

		roles, err := tvm.NewRoles(body)
		if err != nil {
			return nil, xerrors.Errorf("failed to parse file '%s': %v", filename, err)
		}

		res[slug] = roles
	}

	return res, nil
}

func compressAndSignRoles(roles *tvm.Roles, keymap *keys.KeyMap) (preparedRoles, error) {
	compressed, err := compress(roles.GetRaw())
	if err != nil {
		return preparedRoles{}, xerrors.Errorf("failed to compress: %v", err)
	}

	hash := hmac.New(sha256.New, keymap.GetDefaultKey())
	hash.Write(compressed)
	sign := hash.Sum(nil)

	meta := model.Meta{
		Codec:         "brotli",
		DecodedSha256: fmt.Sprintf("%X", sha256.Sum256(roles.GetRaw())),
		DecodedSize:   uint64(len(roles.GetRaw())),
		EncodedHmac:   fmt.Sprintf("%s:%s", keymap.DefaultKeyID, hex.EncodeToString(sign)),
		RevisionExt:   roles.GetMeta().Revision,
	}

	return preparedRoles{
		structuredRoles: roles,
		compressed:      compressed,
		meta:            meta,
	}, nil
}

func compress(blob []byte) ([]byte, error) {
	buffer := new(bytes.Buffer)
	gw := brotli.NewWriterLevel(buffer, 8)

	_, err := gw.Write(blob)
	if err != nil {
		return nil, xerrors.Errorf("failed to compress roles: %w", err)
	}
	if err := gw.Close(); err != nil {
		return nil, xerrors.Errorf("failed to finish compressing of roles: %w", err)
	}

	return buffer.Bytes(), nil
}
