package ytc

import (
	"context"
	"io/ioutil"
	"time"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/infra/daemons/tirole/internal/errs"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type Config struct {
	Cluster        string `json:"cluster"`
	Path           string `json:"path"`
	OAuthTokenFile string `json:"oauth_token_file"`
	Timeout        uint64 `json:"timeout_ms"`
	UseTLS         bool   `json:"use_tls"`
}

type Client struct {
	yc      yt.Client
	timeout yson.Duration
	path    string
	unistat stats
}

type stats struct {
	responseTimings *unistat.TimeStat
	requests        *unistat.SignalDiff
	errs            *unistat.SignalDiff
}

func InitYt(cfg Config) (*Client, error) {
	token, err := ioutil.ReadFile(cfg.OAuthTokenFile)
	if err != nil {
		return nil, xerrors.Errorf("Failed to read oauth token: %w", err)
	}

	cl, err := ythttp.NewClient(&yt.Config{
		Proxy:            cfg.Cluster,
		ProxyRole:        "passport",
		UseTLS:           cfg.UseTLS,
		Token:            string(token),
		Logger:           logger.Log().With(log.Nil("YtClient")).Structured(),
		CompressionCodec: yt.ClientCodecGZIP,
	})
	if err != nil {
		return nil, xerrors.Errorf("Failed to create YT client: %w", err)
	}

	responseTimings, err := unistat.DefaultChunk.CreateTimeStats(
		"yt.response_time",
		unistat.CreateTimeBoundsFromMaxValue(10*time.Second),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create time stats: %w", err)
	}

	return &Client{
		yc:      cl,
		timeout: yson.Duration(time.Duration(cfg.Timeout) * time.Millisecond),
		path:    cfg.Path,
		unistat: stats{
			responseTimings: responseTimings,
			requests:        unistat.DefaultChunk.CreateSignalDiff("yt.requests"),
			errs:            unistat.DefaultChunk.CreateSignalDiff("yt.errors"),
		},
	}, nil
}

func (c *Client) readSingleRow(ctx context.Context, query string, out interface{}) error {
	c.unistat.requests.Inc()

	found := false
	err := ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		found = true
		return ytsimple.ScanRow(reader, out)
	})
	if err != nil {
		return err
	}

	if !found {
		return &errs.NoRolesError{}
	}

	return nil
}
