package tirole

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tirole/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ErrorResponse struct {
	Error string `json:"error"`
}

func (t *Tirole) handleErrorMiddleware() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			err := next(ctx)

			switch err := err.(type) {
			case nil:
				return nil

			case *echo.HTTPError:
				t.unistat.errInvalidParams.Inc()
				ctxlog.Debug(ctx.Request().Context(), logger.Log(), err.Error())
				return sendErrorResponseImpl(ctx, err.Code, fmt.Sprintf("%v", err.Message))

			case *errs.InvalidParamError:
				t.unistat.errInvalidParams.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "%s", err)
				return sendErrorResponseImpl(ctx, http.StatusBadRequest, err.Error())

			case *errs.NoRolesError:
				t.unistat.errNoRoles.Inc()
				text := createErrorMessageForMissingRevision(ctx, err)
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "%s", text)
				return sendErrorResponseImpl(ctx, http.StatusBadRequest, text)

			case *errs.TemporaryError:
				t.unistat.errTmp.Inc()
				text := createErrorMessageForMissingRevision(ctx, err)
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "%s", text)
				return sendErrorResponseImpl(ctx, http.StatusInternalServerError, text)

			case *errs.BadSignError:
				t.unistat.errBadSign.Inc()
				ctxlog.Errorf(ctx.Request().Context(), logger.Log(), "%s", err)
				return sendErrorResponseImpl(ctx, http.StatusInternalServerError, err.Error())

			case *errs.UnauthorizedError:
				t.unistat.errUnauthorized.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "%s. '%s'", err, err.LoggablePart)
				return sendErrorResponseImpl(ctx, http.StatusUnauthorized, err.Error())

			case *errs.AccessDenied:
				t.unistat.errUnauthorized.Inc()
				ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "%s", err)
				return sendErrorResponseImpl(ctx, http.StatusForbidden, err.Error())
			}

			t.unistat.errUnknown.Inc()
			ctxlog.Errorf(ctx.Request().Context(), logger.Log(), "Unexpected error: %s", err)
			return sendErrorResponseImpl(ctx, http.StatusInternalServerError, err.Error())
		}
	}
}

func createErrorMessageForMissingRevision(ctx echo.Context, err error) string {
	st := tvm.ContextServiceTicket(ctx.Request().Context())

	var id string
	if st == nil {
		id = "-"
	} else {
		id = fmt.Sprintf("%d", st.SrcID)
	}

	return fmt.Sprintf("Failed to get revision for tvmid=%s: %s", id, err)
}

func sendErrorResponseImpl(ctx echo.Context, code int, err string) error {
	// TODO: extend
	txt := fmt.Sprintf(
		"%s. request_id=%s",
		err,
		middlewares.ContextReqID(ctx.Request().Context()),
	)

	return ctx.JSON(code, ErrorResponse{Error: txt})
}
